package mekanism.generators.common.tile.reactor;

import io.netty.buffer.ByteBuf;

import java.util.ArrayList;

import mekanism.api.Coord4D;
import mekanism.api.gas.GasRegistry;
import mekanism.api.gas.GasStack;
import mekanism.api.gas.GasTank;
import mekanism.client.SparkleAnimation.INodeChecker;
import mekanism.client.sound.ISoundSource;
import mekanism.common.Mekanism;
import mekanism.common.MekanismFluids;
import mekanism.common.base.IActiveState;
import mekanism.common.base.IHasSound;
import mekanism.common.base.SoundWrapper;
import mekanism.common.config.MekanismConfig.client;
import mekanism.common.network.PacketTileEntity.TileEntityMessage;
import mekanism.common.util.MekanismUtils;
import mekanism.generators.common.FusionReactor;
import net.minecraft.client.audio.ISound.AttenuationType;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.NonNullList;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.Vec3d;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidTank;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class TileEntityReactorController extends TileEntityReactorBlock implements IActiveState, IHasSound, ISoundSource
{
	public static final int MAX_WATER = 100 * Fluid.BUCKET_VOLUME;
	public static final int MAX_STEAM = MAX_WATER * 100;
	public static final int MAX_FUEL = 1 * Fluid.BUCKET_VOLUME;

	public FluidTank waterTank = new FluidTank(MAX_WATER);
	public FluidTank steamTank = new FluidTank(MAX_STEAM);

	public GasTank deuteriumTank = new GasTank(MAX_FUEL);
	public GasTank tritiumTank = new GasTank(MAX_FUEL);

	public GasTank fuelTank = new GasTank(MAX_FUEL);

	public AxisAlignedBB box;
	
	public ResourceLocation soundURL = new ResourceLocation("mekanism", "tile.machine.fusionreactor");
	
	@SideOnly(Side.CLIENT)
	public SoundWrapper sound;

	public double clientTemp = 0;
	public boolean clientBurning = false;

	public TileEntityReactorController()
	{
		super("ReactorController", 1000000000);
		inventory = NonNullList.func_191197_a(1, ItemStack.field_190927_a);
	}

	public boolean isFrame()
	{
		return false;
	}

	public void radiateNeutrons(int neutrons) {} //future impl

	public void formMultiblock(boolean keepBurning)
	{
		if(getReactor() == null)
		{
			setReactor(new FusionReactor(this));
		}
		
		getReactor().formMultiblock(keepBurning);
	}

	public double getPlasmaTemp()
	{
		if(getReactor() == null || !getReactor().isFormed())
		{
			return 0;
		}
		
		return getReactor().getPlasmaTemp();
	}

	public double getCaseTemp()
	{
		if(getReactor() == null || !getReactor().isFormed())
		{
			return 0;
		}
		
		return getReactor().getCaseTemp();
	}

	@Override
	public void onUpdate()
	{
		super.onUpdate();
		
		if(field_145850_b.field_72995_K)
		{
			updateSound();
		}

		if(isFormed())
		{
			getReactor().simulate();
			
			if(!field_145850_b.field_72995_K && (getReactor().isBurning() != clientBurning || Math.abs(getReactor().getPlasmaTemp() - clientTemp) > 1000000))
			{
				Mekanism.packetHandler.sendToAllAround(new TileEntityMessage(Coord4D.get(this), getNetworkedData(new ArrayList())), Coord4D.get(this).getTargetPoint(50D));
				clientBurning = getReactor().isBurning();
				clientTemp = getReactor().getPlasmaTemp();
			}
		}
	}

	@SideOnly(Side.CLIENT)
	public void updateSound()
	{
		if(shouldPlaySound() && getSound().canRestart() && client.enableMachineSounds)
		{
			getSound().reset();
			getSound().play();
		}
	}

	@Override
	public void onChunkUnload()
	{
		super.onChunkUnload();
		
		formMultiblock(true);
	}
	
	@Override
	public void onAdded()
	{
		super.onAdded();
		
		formMultiblock(false);
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound tag)
	{
		super.func_189515_b(tag);

		tag.func_74757_a("formed", isFormed());

		if(isFormed())
		{
			tag.func_74780_a("plasmaTemp", getReactor().getPlasmaTemp());
			tag.func_74780_a("caseTemp", getReactor().getCaseTemp());
			tag.func_74768_a("injectionRate", getReactor().getInjectionRate());
			tag.func_74757_a("burning", getReactor().isBurning());
		}
		else {
			tag.func_74780_a("plasmaTemp", 0);
			tag.func_74780_a("caseTemp", 0);
			tag.func_74768_a("injectionRate", 0);
			tag.func_74757_a("burning", false);
		}

		tag.func_74782_a("fuelTank", fuelTank.write(new NBTTagCompound()));
		tag.func_74782_a("deuteriumTank", deuteriumTank.write(new NBTTagCompound()));
		tag.func_74782_a("tritiumTank", tritiumTank.write(new NBTTagCompound()));
		tag.func_74782_a("waterTank", waterTank.writeToNBT(new NBTTagCompound()));
		tag.func_74782_a("steamTank", steamTank.writeToNBT(new NBTTagCompound()));
		
		return tag;
	}

	@Override
	public void func_145839_a(NBTTagCompound tag)
	{
		super.func_145839_a(tag);

		boolean formed = tag.func_74767_n("formed");

		if(formed)
		{
			setReactor(new FusionReactor(this));
			getReactor().setPlasmaTemp(tag.func_74769_h("plasmaTemp"));
			getReactor().setCaseTemp(tag.func_74769_h("caseTemp"));
			getReactor().setInjectionRate(tag.func_74762_e("injectionRate"));
			getReactor().setBurning(tag.func_74767_n("burning"));
			getReactor().updateTemperatures();
		}

		fuelTank.read(tag.func_74775_l("fuelTank"));
		deuteriumTank.read(tag.func_74775_l("deuteriumTank"));
		tritiumTank.read(tag.func_74775_l("tritiumTank"));
		waterTank.readFromNBT(tag.func_74775_l("waterTank"));
		steamTank.readFromNBT(tag.func_74775_l("steamTank"));
	}

	@Override
	public ArrayList<Object> getNetworkedData(ArrayList<Object> data)
	{
		super.getNetworkedData(data);

		data.add(getReactor() != null && getReactor().isFormed());
		
		if(getReactor() != null)
		{
			data.add(getReactor().getPlasmaTemp());
			data.add(getReactor().getCaseTemp());
			data.add(getReactor().getInjectionRate());
			data.add(getReactor().isBurning());
			data.add(fuelTank.getStored());
			data.add(deuteriumTank.getStored());
			data.add(tritiumTank.getStored());
			data.add(waterTank.getCapacity());
			data.add(waterTank.getFluidAmount());
			data.add(steamTank.getCapacity());
			data.add(steamTank.getFluidAmount());
		}

		return data;
	}

	@Override
	public void handlePacketData(ByteBuf dataStream)
	{
		if(FMLCommonHandler.instance().getEffectiveSide().isServer())
		{
			int type = dataStream.readInt();

			switch(type)
			{
				case 0:
					if(getReactor() != null) getReactor().setInjectionRate(dataStream.readInt());
					break;
			}

			return;
		}

		super.handlePacketData(dataStream);

		if(FMLCommonHandler.instance().getEffectiveSide().isClient())
		{
			boolean formed = dataStream.readBoolean();
			
			if(formed)
			{
				if(getReactor() == null || !((FusionReactor)getReactor()).formed)
				{
					Mekanism.proxy.doGenericSparkle(this, new INodeChecker() {
						@Override
						public boolean isNode(TileEntity tile)
						{
							return tile instanceof TileEntityReactorBlock;
						}
					});
				}
				
				if(getReactor() == null)
				{
					setReactor(new FusionReactor(this));
					MekanismUtils.updateBlock(field_145850_b, func_174877_v());
				}
				
				((FusionReactor)getReactor()).formed = true;
				getReactor().setPlasmaTemp(dataStream.readDouble());
				getReactor().setCaseTemp(dataStream.readDouble());
				getReactor().setInjectionRate(dataStream.readInt());
				getReactor().setBurning(dataStream.readBoolean());
				fuelTank.setGas(new GasStack(MekanismFluids.FusionFuel, dataStream.readInt()));
				deuteriumTank.setGas(new GasStack(MekanismFluids.Deuterium, dataStream.readInt()));
				tritiumTank.setGas(new GasStack(MekanismFluids.Tritium, dataStream.readInt()));
				waterTank.setCapacity(dataStream.readInt());
				waterTank.setFluid(new FluidStack(FluidRegistry.getFluid("water"), dataStream.readInt()));
				steamTank.setCapacity(dataStream.readInt());
				steamTank.setFluid(new FluidStack(FluidRegistry.getFluid("steam"), dataStream.readInt()));
			}
			else if(getReactor() != null)
			{
				setReactor(null);
				MekanismUtils.updateBlock(field_145850_b, func_174877_v());
			}
		}
	}

	public boolean isFormed()
	{
		return getReactor() != null && getReactor().isFormed();
	}

	public boolean isBurning()
	{
		return getActive() && getReactor().isBurning();
	}

	@Override
	public boolean getActive()
	{
		return isFormed();
	}

	@Override
	public void setActive(boolean active)
	{
		if(active == (getReactor() == null))
		{
			setReactor(active ? new FusionReactor(this) : null);
		}
	}

	@Override
	public boolean renderUpdate()
	{
		return true;
	}

	@Override
	public boolean lightUpdate()
	{
		return false;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public AxisAlignedBB getRenderBoundingBox()
	{
		if(box == null)
		{
			box = new AxisAlignedBB(func_174877_v().func_177958_n()-1, func_174877_v().func_177956_o()-3, func_174877_v().func_177952_p()-1, func_174877_v().func_177958_n()+2, func_174877_v().func_177956_o(), func_174877_v().func_177952_p()+2);
		}
		
		return box;
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public SoundWrapper getSound()
	{
		return sound;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public boolean shouldPlaySound()
	{
		return isBurning() && !func_145837_r();
	}

	@Override
	@SideOnly(Side.CLIENT)
	public ResourceLocation getSoundLocation()
	{
		return soundURL;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public float getVolume()
	{
		return 2F;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public float getFrequency()
	{
		return 1F;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public Vec3d getSoundPosition()
	{
		return new Vec3d(func_174877_v()).func_72441_c(0.5, 0.5, 0.5);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public boolean shouldRepeat()
	{
		return true;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public int getRepeatDelay()
	{
		return 0;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public AttenuationType getAttenuation()
	{
		return AttenuationType.LINEAR;
	}

	@Override
	public void func_145829_t()
	{
		super.func_145829_t();

		if(field_145850_b.field_72995_K)
		{
			initSounds();
		}
	}

	@SideOnly(Side.CLIENT)
	public void initSounds()
	{
		sound = new SoundWrapper(this, this);
	}
}
