package mekanism.generators.common.block;

import java.util.Random;

import mekanism.api.energy.IEnergizedItem;
import mekanism.common.Mekanism;
import mekanism.common.base.IActiveState;
import mekanism.common.base.IBoundingBlock;
import mekanism.common.base.ISustainedData;
import mekanism.common.base.ISustainedInventory;
import mekanism.common.base.ISustainedTank;
import mekanism.common.block.states.BlockStateBasic;
import mekanism.common.block.states.BlockStateFacing;
import mekanism.common.config.MekanismConfig.client;
import mekanism.common.multiblock.IMultiblock;
import mekanism.common.security.ISecurityItem;
import mekanism.common.security.ISecurityTile;
import mekanism.common.tile.TileEntityMultiblock;
import mekanism.common.tile.prefab.TileEntityBasicBlock;
import mekanism.common.tile.prefab.TileEntityContainerBlock;
import mekanism.common.tile.prefab.TileEntityElectricBlock;
import mekanism.common.util.MekanismUtils;
import mekanism.common.util.SecurityUtils;
import mekanism.generators.common.GeneratorsBlocks;
import mekanism.generators.common.GeneratorsItems;
import mekanism.generators.common.MekanismGenerators;
import mekanism.generators.common.block.states.BlockStateGenerator;
import mekanism.generators.common.block.states.BlockStateGenerator.GeneratorBlock;
import mekanism.generators.common.block.states.BlockStateGenerator.GeneratorType;
import mekanism.generators.common.tile.TileEntitySolarGenerator;
import mekanism.generators.common.tile.turbine.TileEntityTurbineRotor;
import net.minecraft.block.Block;
import net.minecraft.block.BlockContainer;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import buildcraft.api.tools.IToolWrench;

/**
 * Block class for handling multiple generator block IDs.
 * 0: Heat Generator
 * 1: Solar Generator
 * 3: Hydrogen Generator
 * 4: Bio-Generator
 * 5: Advanced Solar Generator
 * 6: Wind Generator
 * 7: Turbine Rotor
 * 8: Rotational Complex
 * 9: Electromagnetic Coil
 * 10: Turbine Casing
 * 11: Turbine Valve
 * 12: Turbine Vent
 * 13: Saturating Condenser
 * @author AidanBrady
 *
 */
public abstract class BlockGenerator extends BlockContainer
{
	public Random machineRand = new Random();
	
	private static final AxisAlignedBB SOLAR_BOUNDS = new AxisAlignedBB(0.0F, 0.0F, 0.0F, 1.0F, 0.7F, 1.0F);
	private static final AxisAlignedBB ROTOR_BOUNDS = new AxisAlignedBB(0.375F, 0.0F, 0.375F, 0.625F, 1.0F, 0.625F);

	public BlockGenerator()
	{
		super(Material.field_151573_f);
		func_149711_c(3.5F);
		func_149752_b(8F);
		func_149647_a(Mekanism.tabMekanism);
	}
	
	public static BlockGenerator getGeneratorBlock(GeneratorBlock block)
	{
		return new BlockGenerator()
		{
			@Override
			public GeneratorBlock getGeneratorBlock()
			{
				return block;
			}
		};
	}

	public abstract GeneratorBlock getGeneratorBlock();
	
	@Override
	public BlockStateContainer func_180661_e()
	{
		return new BlockStateGenerator(this, getTypeProperty());
	}

	@Override
	public IBlockState func_176203_a(int meta)
	{
		GeneratorType type = GeneratorType.get(getGeneratorBlock(), meta & 0xF);

		return func_176223_P().func_177226_a(getTypeProperty(), type);
	}

	@Override
	public int func_176201_c(IBlockState state)
	{
		GeneratorType type = state.func_177229_b(getTypeProperty());
		return type.meta;
	}
	
	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess worldIn, BlockPos pos)
	{
		TileEntity tile = MekanismUtils.getTileEntitySafe(worldIn, pos);
		
		if(tile instanceof TileEntityBasicBlock && ((TileEntityBasicBlock)tile).facing != null)
		{
			state = state.func_177226_a(BlockStateFacing.facingProperty, ((TileEntityBasicBlock)tile).facing);
		}
		
		if(tile instanceof IActiveState)
		{
			state = state.func_177226_a(BlockStateGenerator.activeProperty, ((IActiveState)tile).getActive());
		}
		
		return state;
	}

	@Override
	public void func_189540_a(IBlockState state, World world, BlockPos pos, Block neighborBlock, BlockPos neighborPos)
	{
		if(!world.field_72995_K)
		{
			TileEntity tileEntity = world.func_175625_s(pos);
			
			if(tileEntity instanceof IMultiblock)
			{
				((IMultiblock)tileEntity).doUpdate();
			}

			if(tileEntity instanceof TileEntityBasicBlock)
			{
				((TileEntityBasicBlock)tileEntity).onNeighborChange(neighborBlock);
			}
		}
	}

	@Override
	public void func_180633_a(World world, BlockPos pos, IBlockState state, EntityLivingBase entityliving, ItemStack itemstack)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);

		int side = MathHelper.func_76128_c((double)(entityliving.field_70177_z * 4.0F / 360.0F) + 0.5D) & 3;
		int height = Math.round(entityliving.field_70125_A);
		int change = 3;

		if(tileEntity.canSetFacing(0) && tileEntity.canSetFacing(1))
		{
			if(height >= 65)
			{
				change = 1;
			}
			else if(height <= -65)
			{
				change = 0;
			}
		}

		if(change != 0 && change != 1)
		{
			switch(side)
			{
				case 0: change = 2; break;
				case 1: change = 5; break;
				case 2: change = 3; break;
				case 3: change = 4; break;
			}
		}

		tileEntity.setFacing((short)change);
		tileEntity.redstone = world.func_175687_A(pos) > 0;

		if(tileEntity instanceof IBoundingBlock)
		{
			((IBoundingBlock)tileEntity).onPlace();
		}
		
		if(!world.field_72995_K && tileEntity instanceof IMultiblock)
		{
			((IMultiblock)tileEntity).doUpdate();
		}
	}

	@Override
	public int getLightValue(IBlockState state, IBlockAccess world, BlockPos pos)
	{
		if(client.enableAmbientLighting)
		{
			TileEntity tileEntity = MekanismUtils.getTileEntitySafe(world, pos);

			if(tileEntity instanceof IActiveState && !(tileEntity instanceof TileEntitySolarGenerator))
			{
				if(((IActiveState)tileEntity).getActive() && ((IActiveState)tileEntity).lightUpdate())
				{
					return client.ambientLightingLevel;
				}
			}
		}

		return 0;
	}


	@Override
	public int func_180651_a(IBlockState state)
	{
		return state.func_177230_c().func_176201_c(state);
	}
	
	@Override
	public float func_180647_a(IBlockState state, EntityPlayer player, World world, BlockPos pos)
	{
		TileEntity tile = world.func_175625_s(pos);
		
		return SecurityUtils.canAccess(player, tile) ? super.func_180647_a(state, player, world, pos) : 0.0F;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_149666_a(Item i, CreativeTabs creativetabs, NonNullList<ItemStack> list)
	{
		for(GeneratorType type : GeneratorType.values())
		{
			if(type.isEnabled())
			{
				list.add(new ItemStack(i, 1, type.meta));
			}
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_180655_c(IBlockState state, World world, BlockPos pos, Random random)
	{
		GeneratorType type = GeneratorType.get(state.func_177230_c(), state.func_177230_c().func_176201_c(state));
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		
		if(MekanismUtils.isActive(world, pos))
		{
			float xRandom = (float)pos.func_177958_n() + 0.5F;
			float yRandom = (float)pos.func_177956_o() + 0.0F + random.nextFloat() * 6.0F / 16.0F;
			float zRandom = (float)pos.func_177952_p() + 0.5F;
			float iRandom = 0.52F;
			float jRandom = random.nextFloat() * 0.6F - 0.3F;

			if(tileEntity.facing == EnumFacing.WEST)
			{
				switch(type)
				{
					case HEAT_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (double)(xRandom + iRandom), (double)yRandom, (double)(zRandom - jRandom), 0.0D, 0.0D, 0.0D);
						world.func_175688_a(EnumParticleTypes.FLAME, (double)(xRandom + iRandom), (double)yRandom, (double)(zRandom - jRandom), 0.0D, 0.0D, 0.0D);
						break;
					case BIO_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, pos.func_177958_n()+.25, pos.func_177956_o()+.2, pos.func_177952_p()+.5, 0.0D, 0.0D, 0.0D);
						break;
					default:
						break;
				}
			}
			else if(tileEntity.facing == EnumFacing.EAST)
			{
				switch(type)
				{
					case HEAT_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (double)(xRandom + iRandom), (double)yRandom + 0.5F, (double)(zRandom - jRandom), 0.0D, 0.0D, 0.0D);
						world.func_175688_a(EnumParticleTypes.FLAME, (double)(xRandom + iRandom), (double)yRandom + 0.5F, (double)(zRandom - jRandom), 0.0D, 0.0D, 0.0D);
						break;
					case BIO_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, pos.func_177958_n()+.75, pos.func_177956_o()+.2, pos.func_177952_p()+.5, 0.0D, 0.0D, 0.0D);
						break;
					default:
						break;
				}
			}
			else if(tileEntity.facing == EnumFacing.NORTH)
			{
				switch(type)
				{
					case HEAT_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (double)(xRandom - jRandom), (double)yRandom + 0.5F, (double)(zRandom - iRandom), 0.0D, 0.0D, 0.0D);
						world.func_175688_a(EnumParticleTypes.FLAME, (double)(xRandom - jRandom), (double)yRandom + 0.5F, (double)(zRandom - iRandom), 0.0D, 0.0D, 0.0D);
						break;
					case BIO_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, pos.func_177958_n()+.5, pos.func_177956_o()+.2, pos.func_177952_p()+.25, 0.0D, 0.0D, 0.0D);
						break;
					default:
						break;
				}
			}
			else if(tileEntity.facing == EnumFacing.SOUTH)
			{
				switch(type)
				{
					case HEAT_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (double)(xRandom - jRandom), (double)yRandom + 0.5F, (double)(zRandom + iRandom), 0.0D, 0.0D, 0.0D);
						world.func_175688_a(EnumParticleTypes.FLAME, (double)(xRandom - jRandom), (double)yRandom + 0.5F, (double)(zRandom + iRandom), 0.0D, 0.0D, 0.0D);
						break;
					case BIO_GENERATOR:
						world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, pos.func_177958_n()+.5, pos.func_177956_o()+.2, pos.func_177952_p()+.75, 0.0D, 0.0D, 0.0D);
						break;
					default:
						break;
				}
			}
		}
	}

	@Override
	public void func_180663_b(World world, BlockPos pos, IBlockState state)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		
		if(!world.field_72995_K && tileEntity instanceof TileEntityTurbineRotor)
		{
			int amount = ((TileEntityTurbineRotor)tileEntity).getHousedBlades();
			
			if(amount > 0)
			{
				float motion = 0.7F;
				double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
				double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
				double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

				EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, new ItemStack(GeneratorsItems.TurbineBlade, amount));

				world.func_72838_d(entityItem);
			}
		}

		if(tileEntity instanceof IBoundingBlock)
		{
			((IBoundingBlock)tileEntity).onBreak();
		}

		super.func_180663_b(world, pos, state);
	}

	@Override
	public boolean func_180639_a(World world, BlockPos pos, IBlockState state, EntityPlayer entityplayer, EnumHand hand, EnumFacing side, float hitX, float hitY, float hitZ)
	{
		if(world.field_72995_K)
		{
			return true;
		}

		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		int metadata = state.func_177230_c().func_176201_c(state);
		ItemStack stack = entityplayer.func_184586_b(hand);

		if(!stack.func_190926_b())
		{
			Item tool = stack.func_77973_b();

			if(MekanismUtils.hasUsableWrench(entityplayer, pos))
			{
				if(SecurityUtils.canAccess(entityplayer, tileEntity))
				{
					if(entityplayer.func_70093_af())
					{
						dismantleBlock(state, world, pos, false);
						
						return true;
					}
	
					if(MekanismUtils.isBCWrench(tool))
					{
						((IToolWrench)tool).wrenchUsed(entityplayer, hand, stack, new RayTraceResult(new Vec3d(hitX, hitY, hitZ), side, pos));
					}
	
					int change = tileEntity.facing.func_176746_e().ordinal();
	
					tileEntity.setFacing((short)change);
					world.func_175685_c(pos, this, true);
				}
				else {
					SecurityUtils.displayNoAccess(entityplayer);
				}
				
				return true;
			}
		}
		
		if(metadata == GeneratorType.TURBINE_CASING.meta || metadata == GeneratorType.TURBINE_VALVE.meta || metadata == GeneratorType.TURBINE_VENT.meta)
		{
			return ((IMultiblock)tileEntity).onActivate(entityplayer, hand, stack);
		}
		
		if(metadata == GeneratorType.TURBINE_ROTOR.meta)
		{
			TileEntityTurbineRotor rod = (TileEntityTurbineRotor)tileEntity;
			
			if(!entityplayer.func_70093_af())
			{
				if(!stack.func_190926_b() && stack.func_77973_b() == GeneratorsItems.TurbineBlade)
				{
					if(!world.field_72995_K && rod.editBlade(true))
					{
						if(!entityplayer.field_71075_bZ.field_75098_d)
						{
							stack.func_190918_g(1);
							
							if(stack.func_190916_E() == 0)
							{
								entityplayer.func_184611_a(hand, ItemStack.field_190927_a);
							}
						}
					}
					
					return true;
				}
			}
			else {
				if(!world.field_72995_K)
				{
					if(stack.func_190926_b())
					{
						if(rod.editBlade(false))
						{
							if(!entityplayer.field_71075_bZ.field_75098_d)
							{
								entityplayer.func_184611_a(hand, new ItemStack(GeneratorsItems.TurbineBlade));
								entityplayer.field_71071_by.func_70296_d();
							}
						}
					}
					else if(stack.func_77973_b() == GeneratorsItems.TurbineBlade)
					{
						if(stack.func_190916_E() < stack.func_77976_d())
						{
							if(rod.editBlade(false))
							{
								if(!entityplayer.field_71075_bZ.field_75098_d)
								{
									stack.func_190917_f(1);
									entityplayer.field_71071_by.func_70296_d();
								}
							}
						}
					}
				}
				
				return true;
			}
			
			return false;
		}
		
		int guiId = GeneratorType.get(getGeneratorBlock(), metadata).guiId;

		if(guiId != -1 && tileEntity != null)
		{
			if(!entityplayer.func_70093_af())
			{
				if(SecurityUtils.canAccess(entityplayer, tileEntity))
				{
					entityplayer.openGui(MekanismGenerators.instance, guiId, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
				}
				else {
					SecurityUtils.displayNoAccess(entityplayer);
				}
				
				return true;
			}
		}

		return false;
	}

	@Override
	public int func_149745_a(Random random)
	{
		return 0;
	}

	@Override
	public TileEntity createTileEntity(World world, IBlockState state)
	{
		int metadata = state.func_177230_c().func_176201_c(state);
		
		if(GeneratorType.get(getGeneratorBlock(), metadata) == null)
		{
			return null;
		}

		return GeneratorType.get(getGeneratorBlock(), metadata).create();
	}

	@Override
	public Item func_180660_a(IBlockState state, Random random, int fortune)
	{
		return null;
	}
	
	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state)
	{
		return EnumBlockRenderType.MODEL;
	}

	@Override
	public boolean func_149662_c(IBlockState state)
	{
		return false;
	}
	
	@Override
	public boolean func_149686_d(IBlockState state)
	{
		return false;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public BlockRenderLayer func_180664_k()
	{
		return BlockRenderLayer.CUTOUT;
	}
	
	/*This method is not used, metadata manipulation is required to create a Tile Entity.*/
	@Override
	public TileEntity func_149915_a(World world, int meta)
	{
		return null;
	}
	
	@Override
	public AxisAlignedBB func_185496_a(IBlockState state, IBlockAccess world, BlockPos pos)
	{
		GeneratorType type = GeneratorType.get(state);

		switch(type)
		{
			case SOLAR_GENERATOR:
				return SOLAR_BOUNDS;
			case TURBINE_ROTOR:
				return ROTOR_BOUNDS;
			default:
				return super.func_185496_a(state, world, pos);
		}
	}

	@Override
	public boolean removedByPlayer(IBlockState state, World world, BlockPos pos, EntityPlayer player, boolean willHarvest)
	{
		if(!player.field_71075_bZ.field_75098_d && !world.field_72995_K && willHarvest)
		{
			float motion = 0.7F;
			double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

			EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, getPickBlock(state, null, world, pos, player));

			world.func_72838_d(entityItem);
		}

		return world.func_175698_g(pos);
	}

	@Override
    public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		ItemStack itemStack = new ItemStack(GeneratorsBlocks.Generator, 1, state.func_177230_c().func_176201_c(state));

		if(itemStack.func_77978_p() == null && !(tileEntity instanceof TileEntityMultiblock))
		{
			itemStack.func_77982_d(new NBTTagCompound());
		}
		
		if(tileEntity == null)
		{
			return ItemStack.field_190927_a;
		}
		
		if(tileEntity instanceof ISecurityTile)
		{
			ISecurityItem securityItem = (ISecurityItem)itemStack.func_77973_b();
			
			if(securityItem.hasSecurity(itemStack))
			{
				securityItem.setOwnerUUID(itemStack, ((ISecurityTile)tileEntity).getSecurity().getOwnerUUID());
				securityItem.setSecurity(itemStack, ((ISecurityTile)tileEntity).getSecurity().getMode());
			}
		}

		if(tileEntity instanceof TileEntityElectricBlock)
		{
			IEnergizedItem electricItem = (IEnergizedItem)itemStack.func_77973_b();
			electricItem.setEnergy(itemStack, ((TileEntityElectricBlock)tileEntity).electricityStored);
		}

		if(tileEntity instanceof TileEntityContainerBlock && ((TileEntityContainerBlock)tileEntity).handleInventory())
		{
			ISustainedInventory inventory = (ISustainedInventory)itemStack.func_77973_b();
			inventory.setInventory(((TileEntityContainerBlock)tileEntity).getInventory(), itemStack);
		}
		
		if(tileEntity instanceof ISustainedData)
		{
			((ISustainedData)tileEntity).writeSustainedData(itemStack);
		}

		if(((ISustainedTank)itemStack.func_77973_b()).hasTank(itemStack))
		{
			if(tileEntity instanceof ISustainedTank)
			{
				if(((ISustainedTank)tileEntity).getFluidStack() != null)
				{
					((ISustainedTank)itemStack.func_77973_b()).setFluidStack(((ISustainedTank)tileEntity).getFluidStack(), itemStack);
				}
			}
		}

		return itemStack;
	}

	public ItemStack dismantleBlock(IBlockState state, World world, BlockPos pos, boolean returnBlock)
	{
		ItemStack itemStack = getPickBlock(state, null, world, pos, null);

		world.func_175698_g(pos);

		if(!returnBlock)
		{
			float motion = 0.7F;
			double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

			EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, itemStack);

			world.func_72838_d(entityItem);
		}

		return itemStack;
	}

	@Override
	public boolean isSideSolid(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side)
	{
		GeneratorType type = GeneratorType.get(getGeneratorBlock(), state.func_177230_c().func_176201_c(state));

		if(type != GeneratorType.SOLAR_GENERATOR && 
				type != GeneratorType.ADVANCED_SOLAR_GENERATOR && 
				type != GeneratorType.WIND_GENERATOR &&
				type != GeneratorType.TURBINE_ROTOR)
		{
			return true;
		}

		return false;
	}

	@Override
	public EnumFacing[] getValidRotations(World world, BlockPos pos)
	{
		TileEntity tile = world.func_175625_s(pos);
		EnumFacing[] valid = new EnumFacing[6];
		
		if(tile instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock basicTile = (TileEntityBasicBlock)tile;
			
			for(EnumFacing dir : EnumFacing.field_82609_l)
			{
				if(basicTile.canSetFacing(dir.ordinal()))
				{
					valid[dir.ordinal()] = dir;
				}
			}
		}
		
		return valid;
	}

	@Override
	public boolean rotateBlock(World world, BlockPos pos, EnumFacing axis)
	{
		TileEntity tile = world.func_175625_s(pos);
		
		if(tile instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock basicTile = (TileEntityBasicBlock)tile;
			
			if(basicTile.canSetFacing(axis.ordinal()))
			{
				basicTile.setFacing((short)axis.ordinal());
				return true;
			}
		}
		
		return false;
	}

	public PropertyEnum<GeneratorType> getTypeProperty()
	{
		return getGeneratorBlock().getProperty();
	}
}
