package mekanism.common.tile.transmitter;

import io.netty.buffer.ByteBuf;

import java.util.ArrayList;
import java.util.Collection;

import mekanism.api.transmitters.TransmissionType;
import mekanism.common.Tier;
import mekanism.common.Tier.BaseTier;
import mekanism.common.Tier.PipeTier;
import mekanism.common.base.FluidHandlerWrapper;
import mekanism.common.base.IFluidHandlerWrapper;
import mekanism.common.block.states.BlockStateTransmitter.TransmitterType;
import mekanism.common.capabilities.CapabilityWrapperManager;
import mekanism.common.transmitters.grid.FluidNetwork;
import mekanism.common.util.CapabilityUtils;
import mekanism.common.util.PipeUtils;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidTank;
import net.minecraftforge.fluids.FluidTankInfo;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;
import net.minecraftforge.fluids.capability.IFluidHandler;

import mekanism.common.tile.transmitter.TileEntitySidedPipe.ConnectionType;

public class TileEntityMechanicalPipe extends TileEntityTransmitter<IFluidHandler, FluidNetwork> implements IFluidHandlerWrapper
{
	public Tier.PipeTier tier = Tier.PipeTier.BASIC;
	
	public float currentScale;

	public FluidTank buffer = new FluidTank(Fluid.BUCKET_VOLUME);

	public FluidStack lastWrite;
	
	@Override
	public BaseTier getBaseTier()
	{
		return tier.getBaseTier();
	}

	@Override
	public void setBaseTier(BaseTier baseTier)
	{
		tier = Tier.PipeTier.get(baseTier);
		buffer.setCapacity(getCapacity());
	}
	
	@Override
	public void func_73660_a()
	{
		if(!func_145831_w().field_72995_K)
		{
            updateShare();
            
			IFluidHandler[] connectedAcceptors = PipeUtils.getConnectedAcceptors(func_174877_v(), func_145831_w());

			for(EnumFacing side : getConnections(ConnectionType.PULL))
			{
				if(connectedAcceptors[side.ordinal()] != null)
				{
					IFluidHandler container = connectedAcceptors[side.ordinal()];

					if(container != null)
					{
						FluidStack received = container.drain(getPullAmount(), false);

						if(received != null && received.amount != 0)
						{
							container.drain(takeFluid(received, true), true);
						}
					}
				}
			}
		}

		super.func_73660_a();
	}

    @Override
    public void updateShare()
    {
        if(getTransmitter().hasTransmitterNetwork() && getTransmitter().getTransmitterNetworkSize() > 0)
        {
            FluidStack last = getSaveShare();

            if((last != null && !(lastWrite != null && lastWrite.amount == last.amount && lastWrite.getFluid() == last.getFluid())) || (last == null && lastWrite != null))
            {
                lastWrite = last;
                func_70296_d();
            }
        }
    }

	private FluidStack getSaveShare()
	{
		if(getTransmitter().hasTransmitterNetwork() && getTransmitter().getTransmitterNetwork().buffer != null)
		{
			int remain = getTransmitter().getTransmitterNetwork().buffer.amount%getTransmitter().getTransmitterNetwork().transmitters.size();
			int toSave = getTransmitter().getTransmitterNetwork().buffer.amount/getTransmitter().getTransmitterNetwork().transmitters.size();

			if(getTransmitter().getTransmitterNetwork().transmitters.iterator().next().equals(getTransmitter()))
			{
				toSave += remain;
			}

			return PipeUtils.copy(getTransmitter().getTransmitterNetwork().buffer, toSave);
		}

		return null;
	}

	@Override
	public void onChunkUnload()
	{
		if(!func_145831_w().field_72995_K && getTransmitter().hasTransmitterNetwork())
		{
			if(lastWrite != null && getTransmitter().getTransmitterNetwork().buffer != null)
			{
				getTransmitter().getTransmitterNetwork().buffer.amount -= lastWrite.amount;

				if(getTransmitter().getTransmitterNetwork().buffer.amount <= 0)
				{
					getTransmitter().getTransmitterNetwork().buffer = null;
				}
			}
		}

		super.onChunkUnload();
	}

	@Override
	public void func_145839_a(NBTTagCompound nbtTags)
	{
		super.func_145839_a(nbtTags);
		
		if(nbtTags.func_74764_b("tier")) tier = Tier.PipeTier.values()[nbtTags.func_74762_e("tier")];
		buffer.setCapacity(getCapacity());

		if(nbtTags.func_74764_b("cacheFluid"))
		{
			buffer.setFluid(FluidStack.loadFluidStackFromNBT(nbtTags.func_74775_l("cacheFluid")));
		}
        else {
            buffer.setFluid(null);
        }
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbtTags)
	{
		super.func_189515_b(nbtTags);

		if(lastWrite != null && lastWrite.amount > 0)
		{
			nbtTags.func_74782_a("cacheFluid", lastWrite.writeToNBT(new NBTTagCompound()));
		}
        else {
            nbtTags.func_82580_o("cacheFluid");
        }

		nbtTags.func_74768_a("tier", tier.ordinal());
		
		return nbtTags;
	}

	@Override
	public TransmissionType getTransmissionType()
	{
		return TransmissionType.FLUID;
	}

	@Override
	public TransmitterType getTransmitterType()
	{ 
		return TransmitterType.MECHANICAL_PIPE; 
	}

	@Override
	public boolean isValidAcceptor(TileEntity acceptor, EnumFacing side)
	{
		return PipeUtils.isValidAcceptorOnSide(acceptor, side);
	}

	@Override
	public FluidNetwork createNewNetwork()
	{
		return new FluidNetwork();
	}

	@Override
	public FluidNetwork createNetworkByMerging(Collection<FluidNetwork> networks)
	{
		return new FluidNetwork(networks);
	}

	@Override
	public int getCapacity()
	{
		return tier.pipeCapacity;
	}

	@Override
	public FluidStack getBuffer()
	{
		return buffer == null ? null : buffer.getFluid();
	}

	@Override
	public void takeShare()
	{
		if(getTransmitter().hasTransmitterNetwork() && getTransmitter().getTransmitterNetwork().buffer != null && lastWrite != null)
		{
			getTransmitter().getTransmitterNetwork().buffer.amount -= lastWrite.amount;
			buffer.setFluid(lastWrite);
		}
	}

	@Override
	public int fill(EnumFacing from, FluidStack resource, boolean doFill)
	{
		if(getConnectionType(from) == ConnectionType.NORMAL)
		{
			return takeFluid(resource, doFill);
		}

		return 0;
	}

	@Override
	public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain)
	{
		return null;
	}

	@Override
	public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain)
	{
		return null;
	}

	@Override
	public boolean canFill(EnumFacing from, Fluid fluid)
	{
		return getConnectionType(from) == ConnectionType.NORMAL;
	}

	@Override
	public boolean canDrain(EnumFacing from, Fluid fluid)
	{
		return false;
	}

	@Override
	public FluidTankInfo[] getTankInfo(EnumFacing from)
	{
		if(from != null && getConnectionType(from) != ConnectionType.NONE)
		{
			return new FluidTankInfo[] {buffer.getInfo()};
		}

		return PipeUtils.EMPTY;
	}

	public int getPullAmount()
	{
		return tier.pipePullAmount;
	}

	@Override
	public IFluidHandler getCachedAcceptor(EnumFacing side)
	{
		TileEntity tile = getCachedTile(side);
		
		if(CapabilityUtils.hasCapability(tile, CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY, side.func_176734_d()))
		{
			return CapabilityUtils.getCapability(tile, CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY, side.func_176734_d());
		}
		
		return null;
	}
	
	public int takeFluid(FluidStack fluid, boolean doEmit)
	{
		if(getTransmitter().hasTransmitterNetwork())
		{
			return getTransmitter().getTransmitterNetwork().emit(fluid, doEmit);
		}
		else {
			return buffer.fill(fluid, doEmit);
		}
	}
	
	@Override
	public boolean upgrade(int tierOrdinal)
	{
		if(tier.ordinal() < BaseTier.ULTIMATE.ordinal() && tierOrdinal == tier.ordinal()+1)
		{
			tier = PipeTier.values()[tier.ordinal()+1];
			
			markDirtyTransmitters();
			sendDesc = true;
			
			return true;
		}
		
		return false;
	}
	
	@Override
	public void handlePacketData(ByteBuf dataStream) throws Exception
	{
		tier = PipeTier.values()[dataStream.readInt()];
		
		super.handlePacketData(dataStream);
	}

	@Override
	public ArrayList<Object> getNetworkedData(ArrayList<Object> data)
	{
		data.add(tier.ordinal());
		
		super.getNetworkedData(data);
		
		return data;
	}
	
	@Override
	public boolean hasCapability(Capability<?> capability, EnumFacing side)
	{
		return capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY || super.hasCapability(capability, side);
	}
	
	public CapabilityWrapperManager manager = new CapabilityWrapperManager(IFluidHandlerWrapper.class, FluidHandlerWrapper.class);

	@Override
	public <T> T getCapability(Capability<T> capability, EnumFacing side)
	{
		if(capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY)
		{
			return (T)manager.getWrapper(this, side);
		}
		
		return super.getCapability(capability, side);
	}
}
