package mekanism.common.tile.transmitter;

import io.netty.buffer.ByteBuf;

import java.util.ArrayList;
import java.util.Collection;

import mekanism.api.Coord4D;
import mekanism.api.EnumColor;
import mekanism.api.Range4D;
import mekanism.api.transmitters.TransmissionType;
import mekanism.common.Mekanism;
import mekanism.common.Tier;
import mekanism.common.Tier.BaseTier;
import mekanism.common.Tier.TransporterTier;
import mekanism.common.base.ILogisticalTransporter;
import mekanism.common.block.property.PropertyColor;
import mekanism.common.block.states.BlockStateTransmitter.TransmitterType;
import mekanism.common.capabilities.Capabilities;
import mekanism.common.content.transporter.PathfinderCache;
import mekanism.common.content.transporter.TransitRequest;
import mekanism.common.content.transporter.TransitRequest.TransitResponse;
import mekanism.common.content.transporter.TransporterStack;
import mekanism.common.integration.multipart.MultipartTileNetworkJoiner;
import mekanism.common.network.PacketTileEntity.TileEntityMessage;
import mekanism.common.transmitters.TransporterImpl;
import mekanism.common.transmitters.grid.InventoryNetwork;
import mekanism.common.util.CapabilityUtils;
import mekanism.common.util.LangUtils;
import mekanism.common.util.MekanismUtils;
import mekanism.common.util.TransporterUtils;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumActionResult;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.text.TextComponentString;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.common.util.Constants.NBT;
import net.minecraftforge.fml.common.FMLCommonHandler;

import mekanism.common.tile.transmitter.TileEntitySidedPipe.ConnectionType;

public class TileEntityLogisticalTransporter extends TileEntityTransmitter<TileEntity, InventoryNetwork>
{
	public Tier.TransporterTier tier = Tier.TransporterTier.BASIC;

	public int pullDelay = 0;
	
	public TileEntityLogisticalTransporter()
	{
		transmitterDelegate = new TransporterImpl(this);
	}
	
	@Override
	public BaseTier getBaseTier()
	{
		return tier.getBaseTier();
	}
	
	@Override
	public void setBaseTier(BaseTier baseTier)
	{
		tier = Tier.TransporterTier.get(baseTier);
	}

	@Override
	public TransmitterType getTransmitterType()
	{
		return TransmitterType.LOGISTICAL_TRANSPORTER;
	}

	@Override
	public TransmissionType getTransmissionType()
	{
		return TransmissionType.ITEM;
	}

	@Override
	public void onWorldSeparate()
	{
		super.onWorldSeparate();
		
		if(!func_145831_w().field_72995_K)
		{
			PathfinderCache.onChanged(new Coord4D(func_174877_v(), func_145831_w()));
		}
	}
	
	@Override
	public TileEntity getCachedAcceptor(EnumFacing side)
	{
		return getCachedTile(side);
	}

	@Override
	public boolean isValidTransmitter(TileEntity tileEntity)
	{
		ILogisticalTransporter transporter = CapabilityUtils.getCapability(tileEntity, Capabilities.LOGISTICAL_TRANSPORTER_CAPABILITY, null);
	
		if(getTransmitter().getColor() == null || transporter.getColor() == null || getTransmitter().getColor() == transporter.getColor())
		{
			return super.isValidTransmitter(tileEntity);
		}
		
		return false;
	}

	@Override
	public boolean isValidAcceptor(TileEntity tile, EnumFacing side)
	{
		return TransporterUtils.isValidAcceptorOnSide(tile, side);
	}
	
	@Override
	public boolean handlesRedstone()
	{
		return false;
	}

	@Override
	public void func_73660_a()
	{
		super.func_73660_a();

		getTransmitter().update();
	}

	public void pullItems()
	{
		if(pullDelay == 0)
		{
			boolean did = false;

			for(EnumFacing side : getConnections(ConnectionType.PULL))
			{
				TileEntity tile = func_145831_w().func_175625_s(func_174877_v().func_177972_a(side));
				TransitRequest request = TransitRequest.getTopStacks(tile, side, tier.pullAmount);
				
				if(!request.isEmpty())
				{
					TransitResponse response = TransporterUtils.insert(tile, getTransmitter(), request, getTransmitter().getColor(), true, 0);
	
					if(!response.isEmpty())
					{
						did = true;
						response.getInvStack(tile, side.func_176734_d()).use(response.stack.func_190916_E());
					}
				}
			}

			if(did)
			{
				pullDelay = 10;
			}
		}
		else {
			pullDelay--;
		}
	}

	@Override
	public void onWorldJoin()
	{
		super.onWorldJoin();

		PathfinderCache.onChanged(new Coord4D(func_174877_v(), func_145831_w()));
	}

	@Override
	public InventoryNetwork createNewNetwork()
	{
		return new InventoryNetwork();
	}

	@Override
	public InventoryNetwork createNetworkByMerging(Collection<InventoryNetwork> networks)
	{
		return new InventoryNetwork(networks);
	}

	@Override
	public void handlePacketData(ByteBuf dataStream) throws Exception
	{
		if(FMLCommonHandler.instance().getSide().isClient())
		{
			int type = dataStream.readInt();
			
			if(type == 0)
			{
				super.handlePacketData(dataStream);
				
				tier = TransporterTier.values()[dataStream.readInt()];
		
				int c = dataStream.readInt();
	
				EnumColor prev = getTransmitter().getColor();
	
				if(c != -1)
				{
					getTransmitter().setColor(TransporterUtils.colors.get(c));
				}
				else {
					getTransmitter().setColor(null);
				}
	
				if(prev != getTransmitter().getColor())
				{
					MekanismUtils.updateBlock(field_145850_b, field_174879_c);
				}
	
				getTransmitter().transit.clear();
	
				int amount = dataStream.readInt();
	
				for(int i = 0; i < amount; i++)
				{
					getTransmitter().transit.add(TransporterStack.readFromPacket(dataStream));
				}
			}
			else if(type == 1)
			{
				boolean kill = dataStream.readBoolean();
				int index = dataStream.readInt();
	
				if(kill)
				{
					getTransmitter().transit.remove(index);
				}
				else {
					TransporterStack stack = TransporterStack.readFromPacket(dataStream);
	
					if(stack.progress == 0)
					{
						stack.progress = 5;
					}
	
					getTransmitter().transit.replace(index, stack);
				}
			}
		}
	}

	@Override
	public ArrayList<Object> getNetworkedData(ArrayList<Object> data)
	{
		data.add(0);
		
		super.getNetworkedData(data);
		
		data.add(tier.ordinal());

		if(getTransmitter().getColor() != null)
		{
			data.add(TransporterUtils.colors.indexOf(getTransmitter().getColor()));
		}
		else {
			data.add(-1);
		}

		data.add(getTransmitter().transit.size());

		for(TransporterStack stack : getTransmitter().transit)
		{
			stack.write(getTransmitter(), data);
		}

		return data;
	}

	public ArrayList<Object> getSyncPacket(TransporterStack stack, boolean kill)
	{
		ArrayList<Object> data = new ArrayList<Object>();
		
		if(Mekanism.hooks.MCMPLoaded)
		{
			MultipartTileNetworkJoiner.addMultipartHeader(this, data, null);
		}

		data.add(1);
		data.add(kill);
		data.add(getTransmitter().transit.indexOf(stack));

		if(!kill)
		{
			stack.write(getTransmitter(), data);
		}

		return data;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbtTags)
	{
		super.func_145839_a(nbtTags);
		
		if(nbtTags.func_74764_b("tier")) tier = TransporterTier.values()[nbtTags.func_74762_e("tier")];

		if(nbtTags.func_74764_b("color"))
		{
			getTransmitter().setColor(TransporterUtils.colors.get(nbtTags.func_74762_e("color")));
		}

		if(nbtTags.func_74764_b("stacks"))
		{
			NBTTagList tagList = nbtTags.func_150295_c("stacks", NBT.TAG_COMPOUND);

			for(int i = 0; i < tagList.func_74745_c(); i++)
			{
				TransporterStack stack = TransporterStack.readFromNBT(tagList.func_150305_b(i));

				getTransmitter().transit.add(stack);
			}
		}
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbtTags)
	{
		super.func_189515_b(nbtTags);
		
		nbtTags.func_74768_a("tier", tier.ordinal());

		if(getTransmitter().getColor() != null)
		{
			nbtTags.func_74768_a("color", TransporterUtils.colors.indexOf(getTransmitter().getColor()));
		}

		NBTTagList stacks = new NBTTagList();

		for(TransporterStack stack : getTransmitter().transit)
		{
			NBTTagCompound tagCompound = new NBTTagCompound();
			stack.write(tagCompound);
			stacks.func_74742_a(tagCompound);
		}

		if(stacks.func_74745_c() != 0)
		{
			nbtTags.func_74782_a("stacks", stacks);
		}
		
		return nbtTags;
	}

	@Override
	protected EnumActionResult onConfigure(EntityPlayer player, int part, EnumFacing side)
	{
		TransporterUtils.incrementColor(getTransmitter());
		onPartChanged(null);
		PathfinderCache.onChanged(new Coord4D(func_174877_v(), func_145831_w()));
		Mekanism.packetHandler.sendToReceivers(new TileEntityMessage(new Coord4D(func_174877_v(), func_145831_w()), getNetworkedData(new ArrayList<>())), new Range4D(new Coord4D(func_174877_v(), func_145831_w())));
		player.func_145747_a(new TextComponentString(EnumColor.DARK_BLUE + "[Mekanism]" + EnumColor.GREY + " " + LangUtils.localize("tooltip.configurator.toggleColor") + ": " + (getTransmitter().getColor() != null ? getTransmitter().getColor().getColoredName() : EnumColor.BLACK + LangUtils.localize("gui.none"))));

		return EnumActionResult.SUCCESS;
	}

	@Override
	public EnumActionResult onRightClick(EntityPlayer player, EnumFacing side)
	{
		super.onRightClick(player, side);
		player.func_145747_a(new TextComponentString(EnumColor.DARK_BLUE + "[Mekanism]" + EnumColor.GREY + " " + LangUtils.localize("tooltip.configurator.viewColor") + ": " + (getTransmitter().getColor() != null ? getTransmitter().getColor().getColoredName() : "None")));
		
		return EnumActionResult.SUCCESS;
	}

	@Override
	public EnumColor getRenderColor()
	{
		return getTransmitter().getColor();
	}

	@Override
	public void onChunkUnload()
	{
		super.onChunkUnload();

		if(!func_145831_w().field_72995_K)
		{
			for(TransporterStack stack : getTransmitter().transit)
			{
				TransporterUtils.drop(getTransmitter(), stack);
			}
		}
	}

	@Override
	public int getCapacity()
	{
		return 0;
	}

	@Override
	public Object getBuffer()
	{
		return null;
	}

	@Override
	public void takeShare() {}

    @Override
    public void updateShare() {}

	@Override
	public TransporterImpl getTransmitter()
	{
		return (TransporterImpl)transmitterDelegate;
	}

	public double getCost()
	{
		return (double)TransporterTier.ULTIMATE.speed / (double)tier.speed;
	}
	
	@Override
	public boolean upgrade(int tierOrdinal)
	{
		if(tier.ordinal() < BaseTier.ULTIMATE.ordinal() && tierOrdinal == tier.ordinal()+1)
		{
			tier = TransporterTier.values()[tier.ordinal()+1];
			
			markDirtyTransmitters();
			sendDesc = true;
			
			return true;
		}
		
		return false;
	}
	
	@Override
	public IBlockState getExtendedState(IBlockState state)
	{
		return ((IExtendedBlockState)super.getExtendedState(state)).withProperty(PropertyColor.INSTANCE, new PropertyColor(getRenderColor()));
	}

	@Override
	public boolean hasCapability(Capability<?> capability, EnumFacing side)
	{
		return capability == Capabilities.LOGISTICAL_TRANSPORTER_CAPABILITY || super.hasCapability(capability, side);
	}

	@Override
	public <T> T getCapability(Capability<T> capability, EnumFacing side)
	{
		if(capability == Capabilities.LOGISTICAL_TRANSPORTER_CAPABILITY)
		{
			return (T)getTransmitter();
		}
		
		return super.getCapability(capability, side);
	}
}
