package mekanism.common.tile;

import io.netty.buffer.ByteBuf;

import java.util.ArrayList;

import mekanism.api.Coord4D;
import mekanism.api.Range4D;
import mekanism.common.Mekanism;
import mekanism.common.PacketHandler;
import mekanism.common.multiblock.IMultiblock;
import mekanism.common.multiblock.MultiblockCache;
import mekanism.common.multiblock.MultiblockManager;
import mekanism.common.multiblock.SynchronizedData;
import mekanism.common.multiblock.UpdateProtocol;
import mekanism.common.network.PacketTileEntity.TileEntityMessage;
import mekanism.common.tile.prefab.TileEntityContainerBlock;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.ItemStackHelper;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public abstract class TileEntityMultiblock<T extends SynchronizedData<T>> extends TileEntityContainerBlock implements IMultiblock<T>
{
	/** The multiblock data for this structure. */
	public T structure;
	
	/** Whether or not to send this multiblock's structure in the next update packet. */
	public boolean sendStructure;

	/** This multiblock's previous "has structure" state. */
	public boolean prevStructure;

	/** Whether or not this multiblock has it's structure, for the client side mechanics. */
	public boolean clientHasStructure;
	
	/** Whether or not this multiblock segment is rendering the structure. */
	public boolean isRendering;
	
	/** This multiblock segment's cached data */
	public MultiblockCache cachedData = getNewCache();
	
	/** This multiblock segment's cached inventory ID */
	public String cachedID = null;
	
	public TileEntityMultiblock(String name)
	{
		super(name);
	}
	
	@Override
	public void onUpdate()
	{
		if(field_145850_b.field_72995_K)
		{
			if(structure == null)
			{
				structure = getNewStructure();
			}

			if(structure != null && clientHasStructure && isRendering)
			{
				if(!prevStructure)
				{
					Mekanism.proxy.doMultiblockSparkle(this);
				}
			}

			prevStructure = clientHasStructure;
		}

		if(playersUsing.size() > 0 && ((field_145850_b.field_72995_K && !clientHasStructure) || (!field_145850_b.field_72995_K && structure == null)))
		{
			for(EntityPlayer player : playersUsing)
			{
				player.func_71053_j();
			}
		}

		if(!field_145850_b.field_72995_K)
		{
			if(structure == null)
			{
				isRendering = false;
				
				if(cachedID != null)
				{
					getManager().updateCache(this);
				}
			}
			
			if(structure == null && ticker == 5)
			{
				doUpdate();
			}

			if(prevStructure == (structure == null))
			{
				if(structure != null && !getSynchronizedData().hasRenderer)
				{
					getSynchronizedData().hasRenderer = true;
					isRendering = true;
					sendStructure = true;
				}

				for(EnumFacing side : EnumFacing.field_82609_l)
				{
					Coord4D obj = Coord4D.get(this).offset(side);
					TileEntity tile = obj.getTileEntity(field_145850_b);

					if(!obj.isAirBlock(field_145850_b) && (tile == null || tile.getClass() != getClass()))
					{
						obj.getBlock(field_145850_b).onNeighborChange(field_145850_b, obj.getPos(), func_174877_v());
					}
				}

				Mekanism.packetHandler.sendToReceivers(new TileEntityMessage(Coord4D.get(this), getNetworkedData(new ArrayList<Object>())), new Range4D(Coord4D.get(this)));
			}

			prevStructure = structure != null;

			if(structure != null)
			{
				getSynchronizedData().didTick = false;

				if(getSynchronizedData().inventoryID != null)
				{
					cachedData.sync(getSynchronizedData());
					cachedID = getSynchronizedData().inventoryID;
					getManager().updateCache(this);
				}
			}
		}
	}
	
	@Override
	public void doUpdate()
	{
		if(!field_145850_b.field_72995_K && (structure == null || !getSynchronizedData().didTick))
		{
			getProtocol().doUpdate();

			if(structure != null)
			{
				getSynchronizedData().didTick = true;
			}
		}
	}
	
	public void sendPacketToRenderer()
	{
		if(structure != null)
		{
			for(Coord4D obj : getSynchronizedData().locations)
			{
				TileEntityMultiblock<T> tileEntity = (TileEntityMultiblock<T>)obj.getTileEntity(field_145850_b);

				if(tileEntity != null && tileEntity.isRendering)
				{
					Mekanism.packetHandler.sendToReceivers(new TileEntityMessage(Coord4D.get(tileEntity), tileEntity.getNetworkedData(new ArrayList<Object>())), new Range4D(Coord4D.get(tileEntity)));
				}
			}
		}
	}
	
	protected abstract T getNewStructure();
	
	public abstract MultiblockCache<T> getNewCache();
	
	protected abstract UpdateProtocol<T> getProtocol();
	
	public abstract MultiblockManager<T> getManager();
	
	@Override
	public ArrayList<Object> getNetworkedData(ArrayList<Object> data)
	{
		super.getNetworkedData(data);

		data.add(isRendering);
		data.add(structure != null);
		
		if(structure != null && isRendering)
		{
			if(sendStructure)
			{
				sendStructure = false;

				data.add(true);

				data.add(getSynchronizedData().volHeight);
				data.add(getSynchronizedData().volWidth);
				data.add(getSynchronizedData().volLength);

				getSynchronizedData().renderLocation.write(data);
				data.add(getSynchronizedData().inventoryID);
			}
			else {
				data.add(false);
			}
		}

		return data;
	}

	@Override
	public void handlePacketData(ByteBuf dataStream)
	{
		super.handlePacketData(dataStream);

		if(FMLCommonHandler.instance().getEffectiveSide().isClient())
		{
			if(structure == null)
			{
				structure = getNewStructure();
			}
	
			isRendering = dataStream.readBoolean();
			clientHasStructure = dataStream.readBoolean();
			
			if(clientHasStructure && isRendering)
			{
				if(dataStream.readBoolean())
				{
					getSynchronizedData().volHeight = dataStream.readInt();
					getSynchronizedData().volWidth = dataStream.readInt();
					getSynchronizedData().volLength = dataStream.readInt();
	
					getSynchronizedData().renderLocation = Coord4D.read(dataStream);
					getSynchronizedData().inventoryID = PacketHandler.readString(dataStream);
				}
			}
		}
	}
	
	@Override
	public void func_145839_a(NBTTagCompound nbtTags)
	{
		super.func_145839_a(nbtTags);

		if(structure == null)
		{
			if(nbtTags.func_74764_b("cachedID"))
			{
				cachedID = nbtTags.func_74779_i("cachedID");
				cachedData.load(nbtTags);
			}
		}
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbtTags)
	{
		super.func_189515_b(nbtTags);

		if(cachedID != null)
		{
			nbtTags.func_74778_a("cachedID", cachedID);
			cachedData.save(nbtTags);
		}
		
		return nbtTags;
	}
	
	@Override
	protected NonNullList<ItemStack> getInventory()
	{
		return structure != null ? structure.getInventory() : null;
	}
	
	@Override
	public boolean onActivate(EntityPlayer player, EnumHand hand, ItemStack stack)
	{
		return false;
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public AxisAlignedBB getRenderBoundingBox()
	{
		return INFINITE_EXTENT_AABB;
	}
	
	@Override
	public boolean handleInventory()
	{
		return false;
	}

	@Override
	public T getSynchronizedData()
	{
		return structure;
	}
}
