package mekanism.common.tile;

import io.netty.buffer.ByteBuf;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import mekanism.api.Chunk3D;
import mekanism.api.Coord4D;
import mekanism.api.Range4D;
import mekanism.common.HashList;
import mekanism.common.Mekanism;
import mekanism.common.Upgrade;
import mekanism.common.base.IActiveState;
import mekanism.common.base.IAdvancedBoundingBlock;
import mekanism.common.base.IRedstoneControl;
import mekanism.common.base.ISustainedData;
import mekanism.common.base.IUpgradeTile;
import mekanism.common.block.states.BlockStateMachine;
import mekanism.common.capabilities.Capabilities;
import mekanism.common.chunkloading.IChunkLoader;
import mekanism.common.config.MekanismConfig.usage;
import mekanism.common.content.miner.MItemStackFilter;
import mekanism.common.content.miner.MOreDictFilter;
import mekanism.common.content.miner.MinerFilter;
import mekanism.common.content.miner.ThreadMinerSearch;
import mekanism.common.content.miner.ThreadMinerSearch.State;
import mekanism.common.content.transporter.InvStack;
import mekanism.common.content.transporter.TransitRequest;
import mekanism.common.content.transporter.TransitRequest.TransitResponse;
import mekanism.common.content.transporter.TransporterManager;
import mekanism.common.inventory.container.ContainerFilter;
import mekanism.common.inventory.container.ContainerNull;
import mekanism.common.network.PacketTileEntity.TileEntityMessage;
import mekanism.common.tile.component.TileComponentChunkLoader;
import mekanism.common.tile.component.TileComponentSecurity;
import mekanism.common.tile.component.TileComponentUpgrade;
import mekanism.common.tile.prefab.TileEntityElectricBlock;
import mekanism.common.util.CapabilityUtils;
import mekanism.common.util.ChargeUtils;
import mekanism.common.util.InventoryUtils;
import mekanism.common.util.ItemDataUtils;
import mekanism.common.util.MekanismUtils;
import mekanism.common.util.MinerUtils;
import mekanism.common.util.TransporterUtils;
import net.minecraft.block.Block;
import net.minecraft.block.BlockBush;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.ChunkPos;
import net.minecraft.world.WorldServer;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.util.Constants.NBT;
import net.minecraftforge.event.world.BlockEvent;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import mekanism.common.base.IRedstoneControl.RedstoneControl;

public class TileEntityDigitalMiner extends TileEntityElectricBlock implements IUpgradeTile, IRedstoneControl, IActiveState, ISustainedData, IChunkLoader, IAdvancedBoundingBlock
{
	public static int[] EJECT_INV;

	public Map<Chunk3D, BitSet> oresToMine = new HashMap<Chunk3D, BitSet>();
	public Map<Integer, MinerFilter> replaceMap = new HashMap<Integer, MinerFilter>();

	public HashList<MinerFilter> filters = new HashList<MinerFilter>();

	public ThreadMinerSearch searcher = new ThreadMinerSearch(this);

	public final double BASE_ENERGY_USAGE = usage.digitalMinerUsage;

	public double energyUsage = usage.digitalMinerUsage;

	public int radius;

	public boolean inverse;

	public int minY = 0;
	public int maxY = 60;

	public boolean doEject = false;
	public boolean doPull = false;
	
	public ItemStack missingStack = ItemStack.field_190927_a;
	
	public int BASE_DELAY = 80;

	public int delay;

	public int delayLength = BASE_DELAY;

	public int clientToMine;

	public boolean isActive;
	public boolean clientActive;

	public boolean silkTouch;

	public boolean running;

	public double prevEnergy;

	public int delayTicks;

	public boolean initCalc = false;

	public int numPowering;
	
	public boolean clientRendering = false;

	/** This machine's current RedstoneControl type. */
	public RedstoneControl controlType = RedstoneControl.DISABLED;

	public TileComponentUpgrade upgradeComponent = new TileComponentUpgrade(this, 28);
	public TileComponentSecurity securityComponent = new TileComponentSecurity(this);
	public TileComponentChunkLoader chunkLoaderComponent = new TileComponentChunkLoader(this);

	public TileEntityDigitalMiner()
	{
		super("DigitalMiner", BlockStateMachine.MachineType.DIGITAL_MINER.baseEnergy);
		inventory = NonNullList.func_191197_a(29, ItemStack.field_190927_a);
		radius = 10;
		
		upgradeComponent.setSupported(Upgrade.ANCHOR);
	}

	@Override
	public void onUpdate()
	{
		super.onUpdate();

		if(getActive())
		{
			for(EntityPlayer player : (HashSet<EntityPlayer>)playersUsing.clone())
			{
				if(player.field_71070_bA instanceof ContainerNull || player.field_71070_bA instanceof ContainerFilter)
				{
					player.func_71053_j();
				}
			}
		}

		if(!field_145850_b.field_72995_K)
		{
			if(!initCalc)
			{
				if(searcher.state == State.FINISHED)
				{
					boolean prevRunning = running;
					
					reset();
					start();
					
					running = prevRunning;
				}

				initCalc = true;
			}

			ChargeUtils.discharge(27, this);

			if(MekanismUtils.canFunction(this) && running && getEnergy() >= getPerTick() && searcher.state == State.FINISHED && oresToMine.size() > 0)
			{
				setActive(true);

				if(delay > 0)
				{
					delay--;
				}

				setEnergy(getEnergy()-getPerTick());

				if(delay == 0)
				{
					boolean did = false;
					
					for(Iterator<Chunk3D> it = oresToMine.keySet().iterator(); it.hasNext();)
					{
						Chunk3D chunk = it.next();
						BitSet set = oresToMine.get(chunk);
						int next = 0;
	
						while(!did)
						{
							int index = set.nextSetBit(next);
							Coord4D coord = getCoordFromIndex(index);
	
							if(index == -1)
							{
								it.remove();
								break;
							}
	
							if(!coord.exists(field_145850_b))
							{
								set.clear(index);
								
								if(set.cardinality() == 0)
								{
									it.remove();
									break;
								}
								
								next = index + 1;
								continue;
							}

							IBlockState state = coord.getBlockState(field_145850_b);
							Block block = state.func_177230_c();
							int meta = block.func_176201_c(state);
	
							if(block == null || coord.isAirBlock(field_145850_b))
							{
								set.clear(index);
								
								if(set.cardinality() == 0)
								{
									it.remove();
									break;
								}
								
								next = index + 1;
								continue;
							}
	
							boolean hasFilter = false;
	
							for(MinerFilter filter : filters)
							{
								if(filter.canFilter(new ItemStack(block, 1, meta)))
								{
									hasFilter = true;
									break;
								}
							}
	
							if(inverse == hasFilter)
							{
								set.clear(index);
								
								if(set.cardinality() == 0)
								{
									it.remove();
									break;
								}
								
								next = index + 1;
								continue;
							}
	
							List<ItemStack> drops = MinerUtils.getDrops(field_145850_b, coord, silkTouch);
	
							if(canInsert(drops) && setReplace(coord, index))
							{
								did = true;
								add(drops);
								set.clear(index);
								
								if(set.cardinality() == 0)
								{
									it.remove();
								}
	
								field_145850_b.func_180498_a(null, 2001, coord.getPos(), Block.func_176210_f(state));
	
								missingStack = ItemStack.field_190927_a;
							}
	
							break;
						}
					}
					
					delay = getDelay();
				}
			}
			else {
				if(prevEnergy >= getEnergy())
				{
					setActive(false);
				}
			}
			
			TransitRequest ejectMap = getEjectItemMap();

			if(doEject && delayTicks == 0 && !ejectMap.isEmpty() && getEjectInv() != null && getEjectTile() != null)
			{
				if(CapabilityUtils.hasCapability(getEjectInv(), Capabilities.LOGISTICAL_TRANSPORTER_CAPABILITY, facing.func_176734_d()))
				{
					TransitResponse response = TransporterUtils.insert(getEjectTile(), CapabilityUtils.getCapability(getEjectInv(), Capabilities.LOGISTICAL_TRANSPORTER_CAPABILITY, facing.func_176734_d()), ejectMap, null, true, 0);

					if(!response.isEmpty())
					{
						response.getInvStack(this, facing.func_176734_d()).use();
					}
				}
				else {
					TransitResponse response = InventoryUtils.putStackInInventory(getEjectInv(), ejectMap, facing.func_176734_d(), false);

					if(!response.isEmpty())
					{
						response.getInvStack(this, facing.func_176734_d()).use();
					}
				}

				delayTicks = 10;
			}
			else if(delayTicks > 0)
			{
				delayTicks--;
			}

			if(playersUsing.size() > 0)
			{
				for(EntityPlayer player : playersUsing)
				{
					Mekanism.packetHandler.sendTo(new TileEntityMessage(Coord4D.get(this), getSmallPacket(new ArrayList<Object>())), (EntityPlayerMP)player);
				}
			}

			prevEnergy = getEnergy();
		}
	}

	public double getPerTick()
	{
		double ret = energyUsage;

		if(silkTouch)
		{
			ret *= 6F;
		}

		int baseRad = Math.max(radius-10, 0);
		ret *= (1 + ((float)baseRad/22F));

		int baseHeight = Math.max((maxY-minY)-60, 0);
		ret *= (1 + ((float)baseHeight/195F));

		return ret;
	}

	public int getDelay()
	{
		return delayLength;
	}

	/*
	 * returns false if unsuccessful
	 */
	public boolean setReplace(Coord4D obj, int index)
	{
		IBlockState state = obj.getBlockState(field_145850_b);
		Block block = state.func_177230_c();
		
		EntityPlayer dummy = Mekanism.proxy.getDummyPlayer((WorldServer)field_145850_b, obj.xCoord, obj.yCoord, obj.zCoord).get();
		BlockEvent.BreakEvent event = new BlockEvent.BreakEvent(field_145850_b, obj.getPos(), state, dummy);
		MinecraftForge.EVENT_BUS.post(event);
		
		if(!event.isCanceled())
		{
			ItemStack stack = getReplace(index);
			 
			if(!stack.func_190926_b())
			{
				field_145850_b.func_180501_a(obj.getPos(), Block.func_149634_a(stack.func_77973_b()).func_176203_a(stack.func_77952_i()), 3);

				IBlockState s = obj.getBlockState(field_145850_b);
				if(s.func_177230_c() instanceof BlockBush && !((BlockBush)s.func_177230_c()).func_180671_f(field_145850_b, obj.getPos(), s))
				{
					s.func_177230_c().func_176226_b(field_145850_b, obj.getPos(), s, 1);
					field_145850_b.func_175698_g(obj.getPos());
				}
				
				return true;
			}
			else {
				MinerFilter filter = replaceMap.get(index);

				if(filter == null || (filter.replaceStack.func_190926_b() || !filter.requireStack))
				{
					field_145850_b.func_175698_g(obj.getPos());
					
					return true;
				}
				
				missingStack = filter.replaceStack;
				
				return false;
			}
		}
		
		return false;
	}

	public ItemStack getReplace(int index)
	{
		MinerFilter filter = replaceMap.get(index);
		
		if(filter == null || filter.replaceStack.func_190926_b())
		{
			return ItemStack.field_190927_a;
		}

		for(int i = 0; i < 27; i++)
		{
			if(!inventory.get(i).func_190926_b() && inventory.get(i).func_77969_a(filter.replaceStack))
			{
				inventory.get(i).func_190918_g(1);

				return MekanismUtils.size(filter.replaceStack, 1);
			}
		}

		if(doPull && getPullInv() != null)
		{
			InvStack stack = InventoryUtils.takeDefinedItem(getPullInv(), EnumFacing.UP, filter.replaceStack.func_77946_l(), 1, 1);

			if(stack != null)
			{
				stack.use();
				return MekanismUtils.size(filter.replaceStack, 1);
			}
		}

		return ItemStack.field_190927_a;
	}

	public NonNullList<ItemStack> copy(NonNullList<ItemStack> stacks)
	{
		NonNullList<ItemStack> toReturn = NonNullList.func_191197_a(stacks.size(), ItemStack.field_190927_a);

		for(int i = 0; i < stacks.size(); i++)
		{
			toReturn.set(i, !stacks.get(i).func_190926_b() ? stacks.get(i).func_77946_l() : ItemStack.field_190927_a);
		}

		return toReturn;
	}

	public TransitRequest getEjectItemMap()
	{
		TransitRequest request = new TransitRequest();
		
		for(int i = 27-1; i >= 0; i--)
		{
			ItemStack stack = inventory.get(i);

			if(!stack.func_190926_b())
			{
				if(isReplaceStack(stack))
				{
					continue;
				}

				if(!request.hasType(stack))
				{
					request.setItem(stack, i);
				}
			}
		}

		return request;
	}

	public boolean canInsert(List<ItemStack> stacks)
	{
		if(stacks.isEmpty())
		{
			return true;
		}

		NonNullList<ItemStack> testInv = copy(inventory);

		int added = 0;

		stacks:
		for(ItemStack stack : stacks)
		{
			stack = stack.func_77946_l();
			
			if(stack.func_190926_b() || stack.func_77973_b() == null)
			{
				continue;
			}
			
			for(int i = 0; i < 27; i++)
			{
				if(!testInv.get(i).func_190926_b() && testInv.get(i).func_77973_b() == null)
				{
					testInv.set(i, ItemStack.field_190927_a);
				}
				
				if(testInv.get(i).func_190926_b())
				{
					testInv.set(i, stack);
					added++;

					continue stacks;
				}
				else if(testInv.get(i).func_77969_a(stack) && testInv.get(i).func_190916_E()+stack.func_190916_E() <= stack.func_77976_d())
				{
					testInv.get(i).func_190917_f(stack.func_190916_E());
					added++;

					continue stacks;
				}
			}
		}

		if(added == stacks.size())
		{
			return true;
		}

		return false;
	}

	public TileEntity getPullInv()
	{
		return Coord4D.get(this).translate(0, 2, 0).getTileEntity(field_145850_b);
	}

	public TileEntity getEjectInv()
	{
		EnumFacing side = facing.func_176734_d();

		return field_145850_b.func_175625_s(func_174877_v().func_177984_a().func_177967_a(side, 2));
	}

	public void add(List<ItemStack> stacks)
	{
		if(stacks.isEmpty())
		{
			return;
		}

		stacks:
		for(ItemStack stack : stacks)
		{
			for(int i = 0; i < 27; i++)
			{
				if(inventory.get(i).func_190926_b())
				{
					inventory.set(i, stack);

					continue stacks;
				}
				else if(inventory.get(i).func_77969_a(stack) && inventory.get(i).func_190916_E()+stack.func_190916_E() <= stack.func_77976_d())
				{
					inventory.get(i).func_190917_f(stack.func_190916_E());

					continue stacks;
				}
			}
		}
	}

	public void start()
	{
		if(searcher.state == State.IDLE)
		{
			searcher.start();
		}

		running = true;

		MekanismUtils.saveChunk(this);
	}

	public void stop()
	{
		if(searcher.state == State.SEARCHING)
		{
			searcher.interrupt();
			reset();

			return;
		}
		else if(searcher.state == State.FINISHED)
		{
			running = false;
		}

		MekanismUtils.saveChunk(this);
	}

	public void reset()
	{
		searcher = new ThreadMinerSearch(this);
		running = false;
		oresToMine.clear();
		replaceMap.clear();
		missingStack = ItemStack.field_190927_a;
		setActive(false);

		MekanismUtils.saveChunk(this);
	}
	
	public boolean isReplaceStack(ItemStack stack)
	{
		for(MinerFilter filter : filters)
		{
			if(!filter.replaceStack.func_190926_b() && filter.replaceStack.func_77969_a(stack))
			{
				return true;
			}
		}
		
		return false;
	}
	
	public int getSize()
	{
		int size = 0;
		
		for(Chunk3D chunk : oresToMine.keySet())
		{
			size += oresToMine.get(chunk).cardinality();
		}
		
		return size;
	}

	@Override
	public void func_174889_b(EntityPlayer player)
	{
		super.func_174889_b(player);

		if(!field_145850_b.field_72995_K)
		{
			Mekanism.packetHandler.sendTo(new TileEntityMessage(Coord4D.get(this), getNetworkedData(new ArrayList())), (EntityPlayerMP)player);
		}
	}

	@Override
	public void func_145839_a(NBTTagCompound nbtTags)
	{
		super.func_145839_a(nbtTags);

		radius = nbtTags.func_74762_e("radius");
		minY = nbtTags.func_74762_e("minY");
		maxY = nbtTags.func_74762_e("maxY");
		doEject = nbtTags.func_74767_n("doEject");
		doPull = nbtTags.func_74767_n("doPull");
		isActive = nbtTags.func_74767_n("isActive");
		running = nbtTags.func_74767_n("running");
		delay = nbtTags.func_74762_e("delay");
		silkTouch = nbtTags.func_74767_n("silkTouch");
		numPowering = nbtTags.func_74762_e("numPowering");
		searcher.state = State.values()[nbtTags.func_74762_e("state")];
		controlType = RedstoneControl.values()[nbtTags.func_74762_e("controlType")];
		inverse = nbtTags.func_74767_n("inverse");

		if(nbtTags.func_74764_b("filters"))
		{
			NBTTagList tagList = nbtTags.func_150295_c("filters", NBT.TAG_COMPOUND);

			for(int i = 0; i < tagList.func_74745_c(); i++)
			{
				filters.add(MinerFilter.readFromNBT(tagList.func_150305_b(i)));
			}
		}
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbtTags)
	{
		super.func_189515_b(nbtTags);

		if(searcher.state == State.SEARCHING)
		{
			reset();
		}

		nbtTags.func_74768_a("radius", radius);
		nbtTags.func_74768_a("minY", minY);
		nbtTags.func_74768_a("maxY", maxY);
		nbtTags.func_74757_a("doEject", doEject);
		nbtTags.func_74757_a("doPull", doPull);
		nbtTags.func_74757_a("isActive", isActive);
		nbtTags.func_74757_a("running", running);
		nbtTags.func_74768_a("delay", delay);
		nbtTags.func_74757_a("silkTouch", silkTouch);
		nbtTags.func_74768_a("numPowering", numPowering);
		nbtTags.func_74768_a("state", searcher.state.ordinal());
		nbtTags.func_74768_a("controlType", controlType.ordinal());
		nbtTags.func_74757_a("inverse", inverse);

		NBTTagList filterTags = new NBTTagList();

		for(MinerFilter filter : filters)
		{
			filterTags.func_74742_a(filter.write(new NBTTagCompound()));
		}

		if(filterTags.func_74745_c() != 0)
		{
			nbtTags.func_74782_a("filters", filterTags);
		}
		
		return nbtTags;
	}

	@Override
	public void handlePacketData(ByteBuf dataStream)
	{
		if(FMLCommonHandler.instance().getEffectiveSide().isServer())
		{
			int type = dataStream.readInt();

			switch(type)
			{
				case 0:
					doEject = !doEject;
					break;
				case 1:
					doPull = !doPull;
					break;
				case 3:
					start();
					break;
				case 4:
					stop();
					break;
				case 5:
					reset();
					break;
				case 6:
					radius = dataStream.readInt();
					break;
				case 7:
					minY = dataStream.readInt();
					break;
				case 8:
					maxY = dataStream.readInt();
					break;
				case 9:
					silkTouch = !silkTouch;
					break;
				case 10:
					inverse = !inverse;
					break;
				case 11:
				{
					// Move filter up
					int filterIndex = dataStream.readInt();
					filters.swap(filterIndex, filterIndex - 1);
					
					for(EntityPlayer player : playersUsing) 
					{
						func_174889_b(player);
					}
					
					break;
				}
				case 12:
				{
					// Move filter down
					int filterIndex = dataStream.readInt();
					filters.swap(filterIndex, filterIndex + 1);
					
					for(EntityPlayer player : playersUsing)
					{
						func_174889_b(player);
					}
					
					break;
				}
			}
			
			MekanismUtils.saveChunk(this);

			for(EntityPlayer player : playersUsing)
			{
				Mekanism.packetHandler.sendTo(new TileEntityMessage(Coord4D.get(this), getGenericPacket(new ArrayList<Object>())), (EntityPlayerMP)player);
			}

			return;
		}

		super.handlePacketData(dataStream);

		if(FMLCommonHandler.instance().getEffectiveSide().isClient())
		{
			int type = dataStream.readInt();
	
			if(type == 0)
			{
				radius = dataStream.readInt();
				minY = dataStream.readInt();
				maxY = dataStream.readInt();
				doEject = dataStream.readBoolean();
				doPull = dataStream.readBoolean();
				clientActive = dataStream.readBoolean();
				running = dataStream.readBoolean();
				silkTouch = dataStream.readBoolean();
				numPowering = dataStream.readInt();
				searcher.state = State.values()[dataStream.readInt()];
				clientToMine = dataStream.readInt();
				controlType = RedstoneControl.values()[dataStream.readInt()];
				inverse = dataStream.readBoolean();
				
				if(dataStream.readBoolean())
				{
					missingStack = new ItemStack(Item.func_150899_d(dataStream.readInt()), 1, dataStream.readInt());
				}
				else {
					missingStack = ItemStack.field_190927_a;
				}
	
				filters.clear();
	
				int amount = dataStream.readInt();
	
				for(int i = 0; i < amount; i++)
				{
					filters.add(MinerFilter.readFromPacket(dataStream));
				}
			}
			else if(type == 1)
			{
				radius = dataStream.readInt();
				minY = dataStream.readInt();
				maxY = dataStream.readInt();
				doEject = dataStream.readBoolean();
				doPull = dataStream.readBoolean();
				clientActive = dataStream.readBoolean();
				running = dataStream.readBoolean();
				silkTouch = dataStream.readBoolean();
				numPowering = dataStream.readInt();
				searcher.state = State.values()[dataStream.readInt()];
				clientToMine = dataStream.readInt();
				controlType = RedstoneControl.values()[dataStream.readInt()];
				inverse = dataStream.readBoolean();
				
				if(dataStream.readBoolean())
				{
					missingStack = new ItemStack(Item.func_150899_d(dataStream.readInt()), 1, dataStream.readInt());
				}
				else {
					missingStack = ItemStack.field_190927_a;
				}
			}
			else if(type == 2)
			{
				filters.clear();
	
				int amount = dataStream.readInt();
	
				for(int i = 0; i < amount; i++)
				{
					filters.add(MinerFilter.readFromPacket(dataStream));
				}
			}
			else if(type == 3)
			{
				clientActive = dataStream.readBoolean();
				running = dataStream.readBoolean();
				clientToMine = dataStream.readInt();
				
				if(dataStream.readBoolean())
				{
					missingStack = new ItemStack(Item.func_150899_d(dataStream.readInt()), 1, dataStream.readInt());
				}
				else {
					missingStack = ItemStack.field_190927_a;
				}
			}
			
			if(clientActive != isActive)
			{
				isActive = clientActive;
				MekanismUtils.updateBlock(field_145850_b, func_174877_v());
			}
		}
	}

	@Override
	public ArrayList<Object> getNetworkedData(ArrayList<Object> data)
	{
		super.getNetworkedData(data);

		data.add(0);

		data.add(radius);
		data.add(minY);
		data.add(maxY);
		data.add(doEject);
		data.add(doPull);
		data.add(isActive);
		data.add(running);
		data.add(silkTouch);
		data.add(numPowering);
		data.add(searcher.state.ordinal());
		
		if(searcher.state == State.SEARCHING)
		{
			data.add(searcher.found);
		}
		else {
			data.add(getSize());
		}

		data.add(controlType.ordinal());
		data.add(inverse);
		
		if(!missingStack.func_190926_b())
		{
			data.add(true);
			data.add(MekanismUtils.getID(missingStack));
			data.add(missingStack.func_77952_i());
		}
		else {
			data.add(false);
		}

		data.add(filters.size());

		for(MinerFilter filter : filters)
		{
			filter.write(data);
		}

		return data;
	}

	public ArrayList<Object> getSmallPacket(ArrayList<Object> data)
	{
		super.getNetworkedData(data);

		data.add(3);

		data.add(isActive);
		data.add(running);

		if(searcher.state == State.SEARCHING)
		{
			data.add(searcher.found);
		}
		else {
			data.add(getSize());
		}
		
		if(!missingStack.func_190926_b())
		{
			data.add(true);
			data.add(MekanismUtils.getID(missingStack));
			data.add(missingStack.func_77952_i());
		}
		else {
			data.add(false);
		}

		return data;
	}

	public ArrayList<Object> getGenericPacket(ArrayList<Object> data)
	{
		super.getNetworkedData(data);

		data.add(1);

		data.add(radius);
		data.add(minY);
		data.add(maxY);
		data.add(doEject);
		data.add(doPull);
		data.add(isActive);
		data.add(running);
		data.add(silkTouch);
		data.add(numPowering);
		data.add(searcher.state.ordinal());

		if(searcher.state == State.SEARCHING)
		{
			data.add(searcher.found);
		}
		else {
			data.add(getSize());
		}

		data.add(controlType.ordinal());
		data.add(inverse);
		
		if(!missingStack.func_190926_b())
		{
			data.add(true);
			data.add(MekanismUtils.getID(missingStack));
			data.add(missingStack.func_77952_i());
		}
		else {
			data.add(false);
		}

		return data;
	}

	public ArrayList getFilterPacket(ArrayList<Object> data)
	{
		super.getNetworkedData(data);

		data.add(2);

		data.add(filters.size());

		for(MinerFilter filter : filters)
		{
			filter.write(data);
		}

		return data;
	}

	public int getTotalSize()
	{
		return getDiameter()*getDiameter()*(maxY-minY+1);
	}

	public int getDiameter()
	{
		return (radius*2)+1;
	}

	public Coord4D getStartingCoord()
	{
		return new Coord4D(func_174877_v().func_177958_n()-radius, minY, func_174877_v().func_177952_p()-radius, field_145850_b.field_73011_w.getDimension());
	}

	public Coord4D getCoordFromIndex(int index)
	{
		int diameter = getDiameter();
		Coord4D start = getStartingCoord();

		int x = start.xCoord+index%diameter;
		int y = start.yCoord+(index/diameter/diameter);
		int z = start.zCoord+(index/diameter)%diameter;

		return new Coord4D(x, y, z, field_145850_b.field_73011_w.getDimension());
	}

	@Override
	public boolean isPowered()
	{
		return redstone || numPowering > 0;
	}

	@Override
	public boolean canPulse()
	{
		return false;
	}

	@Override
	public RedstoneControl getControlType()
	{
		return controlType;
	}

	@Override
	public void setControlType(RedstoneControl type)
	{
		controlType = type;
		MekanismUtils.saveChunk(this);
	}

	@Override
	public TileComponentUpgrade getComponent()
	{
		return upgradeComponent;
	}

	@Override
	public void setActive(boolean active)
	{
		isActive = active;

		if(clientActive != active)
		{
			Mekanism.packetHandler.sendToReceivers(new TileEntityMessage(Coord4D.get(this), getNetworkedData(new ArrayList<Object>())), new Range4D(Coord4D.get(this)));

			clientActive = active;
		}
	}

	@Override
	public boolean getActive()
	{
		return isActive;
	}

	@Override
	public boolean renderUpdate()
	{
		return false;
	}

	@Override
	public boolean lightUpdate()
	{
		return false;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public AxisAlignedBB getRenderBoundingBox()
	{
		return INFINITE_EXTENT_AABB;
	}

	@Override
	public void onPlace()
	{
		for(int x = -1; x <= +1; x++)
		{
			for(int y = 0; y <= +1; y++)
			{
				for(int z = -1; z <= +1; z++)
				{
					if(x == 0 && y == 0 && z == 0)
					{
						continue;
					}

					BlockPos pos1 = func_174877_v().func_177982_a(x, y, z);
					MekanismUtils.makeAdvancedBoundingBlock(field_145850_b, pos1, Coord4D.get(this));
		            field_145850_b.func_175685_c(pos1, func_145838_q(), true);
				}
			}
		}
	}

	@Override
	public boolean canSetFacing(int side)
	{
		return side != 0 && side != 1;
	}

	@Override
	public void onBreak()
	{
		for(int x = -1; x <= +1; x++)
		{
			for(int y = 0; y <= +1; y++)
			{
				for(int z = -1; z <= +1; z++)
				{
					field_145850_b.func_175698_g(func_174877_v().func_177982_a(x, y, z));
				}
			}
		}
	}

	@Override
	public int[] func_180463_a(EnumFacing side)
	{
		return InventoryUtils.EMPTY;
	}

	public TileEntity getEjectTile()
	{
		EnumFacing side = facing.func_176734_d();
		return field_145850_b.func_175625_s(func_174877_v().func_177984_a().func_177972_a(side));
	}

	@Override
	public int[] getBoundSlots(BlockPos location, EnumFacing side)
	{
		EnumFacing dir = facing.func_176734_d();

		BlockPos pull = func_174877_v().func_177984_a();
		BlockPos eject = pull.func_177972_a(dir);
		
		if((location.equals(eject) && side == dir) || (location.equals(pull) && side == EnumFacing.UP))
		{
			if(EJECT_INV == null)
			{
				EJECT_INV = new int[27];

				for(int i = 0; i < EJECT_INV.length; i++)
				{
					EJECT_INV[i] = i;
				}
			}

			return EJECT_INV;
		}

		return InventoryUtils.EMPTY;
	}

	@Override
	public boolean canBoundInsert(BlockPos location, int i, ItemStack itemstack)
	{
		EnumFacing side = facing.func_176734_d();

		BlockPos pull = func_174877_v().func_177984_a();
		BlockPos eject = pull.func_177972_a(side);

		if(location.equals(eject))
		{
			return false;
		}
		else if(location.equals(pull))
		{
			if(!itemstack.func_190926_b() && isReplaceStack(itemstack))
			{
				return true;
			}
		}

		return false;
	}

	@Override
	public boolean canBoundExtract(BlockPos location, int i, ItemStack itemstack, EnumFacing dir)
	{
		EnumFacing side = facing.func_176734_d();

		BlockPos pull = func_174877_v().func_177984_a();
		BlockPos eject = pull.func_177972_a(side);

		if(location.equals(eject))
		{
			if(!itemstack.func_190926_b() && isReplaceStack(itemstack))
			{
				return false;
			}

			return true;
		}
		else if(location.equals(pull))
		{
			return false;
		}

		return false;
	}

	@Override
	public void onPower()
	{
		numPowering++;
	}

	@Override
	public void onNoPower()
	{
		numPowering--;
	}

	public String[] methods = {"setRadius", "setMin", "setMax", "addFilter", "removeFilter", "addOreFilter", "removeOreFilter", "reset", "start", "stop", "getToMine"};

	@Override
	public String[] getMethods()
	{
		return methods;
	}

	@Override
	public Object[] invoke(int method, Object[] arguments) throws Exception
	{
		if(method == 0)
		{
			if(arguments.length != 1 || !(arguments[0] instanceof Double))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			radius = ((Double)arguments[0]).intValue();
		}
		else if(method == 1)
		{
			if(arguments.length != 1 || !(arguments[0] instanceof Double))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			minY = ((Double)arguments[0]).intValue();
		}
		else if(method == 2)
		{
			if(arguments.length != 1 || !(arguments[0] instanceof Double))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			maxY = ((Double)arguments[0]).intValue();
		}
		else if(method == 3)
		{
			if(arguments.length < 1 || !(arguments[0] instanceof Double))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			int id = ((Double)arguments[0]).intValue();
			int meta = 0;

			if(arguments.length > 1)
			{
				if(arguments[1] instanceof Double)
				{
					meta = ((Double)arguments[1]).intValue();
				}
			}

			filters.add(new MItemStackFilter(new ItemStack(Item.func_150899_d(id), 1, meta)));
			
			return new Object[] {"Added filter."};
		}
		else if(method == 4)
		{
			if(arguments.length < 1 || !(arguments[0] instanceof Double))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			int id = ((Double)arguments[0]).intValue();
			Iterator<MinerFilter> iter = filters.iterator();

			while(iter.hasNext())
			{
				MinerFilter filter = iter.next();

				if(filter instanceof MItemStackFilter)
				{
					if(MekanismUtils.getID(((MItemStackFilter)filter).itemType) == id)
					{
						iter.remove();
						return new Object[] {"Removed filter."};
					}
				}
			}
			
			return new Object[] {"Couldn't find filter."};
		}
		else if(method == 5)
		{
			if(arguments.length < 1 || !(arguments[0] instanceof String))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			String ore = (String)arguments[0];
			MOreDictFilter filter = new MOreDictFilter();

			filter.oreDictName = ore;
			filters.add(filter);
			
			return new Object[] {"Added filter."};
		}
		else if(method == 6)
		{
			if(arguments.length < 1 || !(arguments[0] instanceof String))
			{
				return new Object[] {"Invalid parameters."};
			}
			
			String ore = (String)arguments[0];
			Iterator<MinerFilter> iter = filters.iterator();

			while(iter.hasNext())
			{
				MinerFilter filter = iter.next();

				if(filter instanceof MOreDictFilter)
				{
					if(((MOreDictFilter)filter).oreDictName.equals(ore))
					{
						iter.remove();
						return new Object[] {"Removed filter."};
					}
				}
			}
			
			return new Object[] {"Couldn't find filter."};
		}
		else if(method == 7)
		{
			reset();
			return new Object[] {"Reset miner."};
		}
		else if(method == 8)
		{
			start();
			return new Object[] {"Started miner."};
		}
		else if(method == 9)
		{
			stop();
			return new Object[] {"Stopped miner."};
		}
		else if(method == 10)
		{
			return new Object[] {searcher != null ? searcher.found : 0};
		}

		for(EntityPlayer player : playersUsing)
		{
			Mekanism.packetHandler.sendTo(new TileEntityMessage(Coord4D.get(this), getGenericPacket(new ArrayList<Object>())), (EntityPlayerMP)player);
		}
		
		return null;
	}

	@Override
	public NBTTagCompound getConfigurationData(NBTTagCompound nbtTags)
	{
		nbtTags.func_74768_a("radius", radius);
		nbtTags.func_74768_a("minY", minY);
		nbtTags.func_74768_a("maxY", maxY);
		nbtTags.func_74757_a("doEject", doEject);
		nbtTags.func_74757_a("doPull", doPull);
		nbtTags.func_74757_a("silkTouch", silkTouch);
		nbtTags.func_74757_a("inverse", inverse);

		NBTTagList filterTags = new NBTTagList();

		for(MinerFilter filter : filters)
		{
			filterTags.func_74742_a(filter.write(new NBTTagCompound()));
		}

		if(filterTags.func_74745_c() != 0)
		{
			nbtTags.func_74782_a("filters", filterTags);
		}
		
		return nbtTags;
	}

	@Override
	public void setConfigurationData(NBTTagCompound nbtTags)
	{
		radius = nbtTags.func_74762_e("radius");
		minY = nbtTags.func_74762_e("minY");
		maxY = nbtTags.func_74762_e("maxY");
		doEject = nbtTags.func_74767_n("doEject");
		doPull = nbtTags.func_74767_n("doPull");
		silkTouch = nbtTags.func_74767_n("silkTouch");
		inverse = nbtTags.func_74767_n("inverse");

		if(nbtTags.func_74764_b("filters"))
		{
			NBTTagList tagList = nbtTags.func_150295_c("filters", NBT.TAG_COMPOUND);

			for(int i = 0; i < tagList.func_74745_c(); i++)
			{
				filters.add(MinerFilter.readFromNBT(tagList.func_150305_b(i)));
			}
		}
	}

	@Override
	public String getDataType()
	{
		return func_145838_q().func_149739_a() + "." + fullName + ".name";
	}
	
	public void writeSustainedData(ItemStack itemStack) 
	{
		ItemDataUtils.setBoolean(itemStack, "hasMinerConfig", true);

		ItemDataUtils.setInt(itemStack, "radius", radius);
		ItemDataUtils.setInt(itemStack, "minY", minY);
		ItemDataUtils.setInt(itemStack, "maxY", maxY);
		ItemDataUtils.setBoolean(itemStack, "doEject", doEject);
		ItemDataUtils.setBoolean(itemStack, "doPull", doPull);
		ItemDataUtils.setBoolean(itemStack, "silkTouch", silkTouch);
		ItemDataUtils.setBoolean(itemStack, "inverse", inverse);

		NBTTagList filterTags = new NBTTagList();

		for(MinerFilter filter : filters)
		{
			filterTags.func_74742_a(filter.write(new NBTTagCompound()));
		}

		if(filterTags.func_74745_c() != 0)
		{
			ItemDataUtils.setList(itemStack, "filters", filterTags);
		}
	}

	@Override
	public void readSustainedData(ItemStack itemStack)
	{
		if(ItemDataUtils.hasData(itemStack, "hasMinerConfig"))
		{
			radius = ItemDataUtils.getInt(itemStack, "radius");
			minY = ItemDataUtils.getInt(itemStack, "minY");
			maxY = ItemDataUtils.getInt(itemStack, "maxY");
			doEject = ItemDataUtils.getBoolean(itemStack, "doEject");
			doPull = ItemDataUtils.getBoolean(itemStack, "doPull");
			silkTouch = ItemDataUtils.getBoolean(itemStack, "silkTouch");
			inverse = ItemDataUtils.getBoolean(itemStack, "inverse");

			if(ItemDataUtils.hasData(itemStack, "filters"))
			{
				NBTTagList tagList = ItemDataUtils.getList(itemStack, "filters");

				for(int i = 0; i < tagList.func_74745_c(); i++)
				{
					filters.add(MinerFilter.readFromNBT((NBTTagCompound)tagList.func_150305_b(i)));
				}
			}
		}
	}

	@Override
	public void recalculateUpgradables(Upgrade upgrade)
	{
		super.recalculateUpgradables(upgrade);

		switch(upgrade)
		{
			case SPEED:
				delayLength = MekanismUtils.getTicks(this, BASE_DELAY);
			case ENERGY:
				energyUsage = MekanismUtils.getEnergyPerTick(this, BASE_ENERGY_USAGE);
				maxEnergy = MekanismUtils.getMaxEnergy(this, BASE_MAX_ENERGY);
				setEnergy(Math.min(getMaxEnergy(), getEnergy()));
			default:
				break;
		}
	}
	
	@Override
	public boolean canBoundReceiveEnergy(BlockPos coord, EnumFacing side)
	{
		EnumFacing left = MekanismUtils.getLeft(facing);
		EnumFacing right = MekanismUtils.getRight(facing);
		
		if(coord.equals(func_174877_v().func_177972_a(left)))
		{
			return side == left;
		}
		else if(coord.equals(func_174877_v().func_177972_a(right)))
		{
			return side == right;
		}
		
		return false;
	}
	
	@Override
	public boolean sideIsConsumer(EnumFacing side)
	{
		return side == MekanismUtils.getLeft(facing) || side == MekanismUtils.getRight(facing) || side == EnumFacing.DOWN;
	}

	@Override
	public TileComponentSecurity getSecurity() 
	{
		return securityComponent;
	}
	
	@Override
	public boolean hasCapability(Capability<?> capability, EnumFacing side)
	{
		return capability == Capabilities.CONFIG_CARD_CAPABILITY || capability == Capabilities.SPECIAL_CONFIG_DATA_CAPABILITY 
				|| super.hasCapability(capability, side);
	}

	@Override
	public <T> T getCapability(Capability<T> capability, EnumFacing side)
	{
		if(capability == Capabilities.CONFIG_CARD_CAPABILITY || capability == Capabilities.SPECIAL_CONFIG_DATA_CAPABILITY)
		{
			return (T)this;
		}
		
		return super.getCapability(capability, side);
	}
	
	@Override
	public TileComponentChunkLoader getChunkLoader()
	{
		return chunkLoaderComponent;
	}
	
	@Override
	public Set<ChunkPos> getChunkSet()
	{
		return new Range4D(Coord4D.get(this)).expandFromCenter(radius).getIntersectingChunks().stream().map(t -> t.getPos()).collect(Collectors.toSet());
	}
}
