package mekanism.common.recipe;

import java.util.HashMap;

import mekanism.common.Mekanism;
import mekanism.common.MekanismItems;
import mekanism.common.util.InventoryUtils;
import mekanism.common.util.RecipeUtils;
import net.minecraft.block.Block;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.NonNullList;
import net.minecraft.world.World;
import net.minecraftforge.common.ForgeHooks;
import net.minecraftforge.common.util.Constants.NBT;
import net.minecraftforge.oredict.OreDictionary;

/**
 * Code originally from Eloraam and her work on the Ore Dictionary.  Cleaned up and modified to work well with energized items.
 * @author Eloraam, aidancbrady
 *
 */
public class ShapedMekanismRecipe implements IRecipe
{
	private static final int MAX_CRAFT_GRID_WIDTH = 3;
	private static final int MAX_CRAFT_GRID_HEIGHT = 3;

	private ItemStack output = ItemStack.field_190927_a;
	private Object[] input = null;

	public int width = 0;
	public int height = 0;

	private boolean mirrored = true;
	
	public ShapedMekanismRecipe(ItemStack result, Object... recipe)
	{
		output = result.func_77946_l();

		String shape = "";
		int idx = 0;

		if(recipe[idx] instanceof Boolean)
		{
			mirrored = (Boolean)recipe[idx];

			if(recipe[idx+1] instanceof Object[])
			{
				recipe = (Object[])recipe[idx+1];
			}
			else {
				idx = 1;
			}
		}

		if(recipe[idx] instanceof String[])
		{
			String[] parts = ((String[])recipe[idx++]);

			for(String s : parts)
			{
				width = s.length();
				shape += s;
			}

			height = parts.length;
		}
		else {
			while(recipe[idx] instanceof String)
			{
				String s = (String)recipe[idx++];
				shape += s;
				width = s.length();
				height++;
			}
		}

		if(width * height != shape.length())
		{
			String ret = "Invalid shaped Mekanism recipe: ";

			for(Object tmp :  recipe)
			{
				ret += tmp + ", ";
			}

			ret += output;

			throw new RuntimeException(ret);
		}

		HashMap<Character, Object> itemMap = new HashMap<Character, Object>();

		for(; idx < recipe.length; idx += 2)
		{
			Character chr = (Character)recipe[idx];
			Object in = recipe[idx + 1];

			if(in instanceof ItemStack)
			{
				itemMap.put(chr, ((ItemStack)in).func_77946_l());
			}
			else if(in instanceof Item)
			{
				itemMap.put(chr, new ItemStack((Item)in));
			}
			else if(in instanceof Block)
			{
				itemMap.put(chr, new ItemStack((Block)in, 1, OreDictionary.WILDCARD_VALUE));
			}
			else if(in instanceof String)
			{
				itemMap.put(chr, OreDictionary.getOres((String)in));
			}
			else {
				String ret = "Invalid shaped Mekanism recipe: ";

				for(Object tmp :  recipe)
				{
					ret += tmp + ", ";
				}

				ret += output;
				throw new RuntimeException(ret);
			}
		}

		input = new Object[width * height];
		int x = 0;

		for(char chr : shape.toCharArray())
		{
			input[x++] = itemMap.get(chr);
		}
	}

	@Override
	public ItemStack func_77572_b(InventoryCrafting inv)
	{
		return RecipeUtils.getCraftingResult(inv, output.func_77946_l());
	}

	@Override
	public int func_77570_a()
	{
		return input.length;
	}

	@Override
	public ItemStack func_77571_b()
	{
		return output;
	}

	@Override
	public NonNullList<ItemStack> func_179532_b(InventoryCrafting inv)
	{
		return ForgeHooks.defaultRecipeGetRemainingItems(inv);
	}

	@Override
	public boolean func_77569_a(InventoryCrafting inv, World world)
	{
		for(int x = 0; x <= MAX_CRAFT_GRID_WIDTH - width; x++)
		{
			for(int y = 0; y <= MAX_CRAFT_GRID_HEIGHT - height; ++y)
			{
				if(checkMatch(inv, x, y, true))
				{
					return true;
				}

				if(mirrored && checkMatch(inv, x, y, false))
				{
					return true;
				}
			}
		}

		return false;
	}

	private boolean checkMatch(InventoryCrafting inv, int startX, int startY, boolean mirror)
	{
		for(int x = 0; x < MAX_CRAFT_GRID_WIDTH; x++)
		{
			for(int y = 0; y < MAX_CRAFT_GRID_HEIGHT; y++)
			{
				int subX = x - startX;
				int subY = y - startY;
				Object target = null;

				if(subX >= 0 && subY >= 0 && subX < width && subY < height)
				{
					if(mirror)
					{
						target = input[width - subX - 1 + subY * width];
					}
					else {
						target = input[subX + subY * width];
					}
				}

				ItemStack slot = inv.func_70463_b(x, y);

				if(target instanceof ItemStack)
				{
					if(!RecipeUtils.areItemsEqualForCrafting((ItemStack)target, slot))
					{
						return false;
					}
				}
				else if(target instanceof Iterable)
				{
					boolean matched = false;

					for(ItemStack item : (Iterable<ItemStack>)target)
					{
						matched = matched || RecipeUtils.areItemsEqualForCrafting(item, slot);
					}

					if(!matched)
					{
						return false;
					}
				}
				else if(target == null && !slot.func_190926_b())
				{
					return false;
				}
			}
		}

		return true;
	}

	public ShapedMekanismRecipe setMirrored(boolean mirror)
	{
		mirrored = mirror;
		return this;
	}

	public Object[] getInput()
	{
		return input;
	}
	
	public static ShapedMekanismRecipe create(NBTTagCompound nbtTags)
	{
		if(!nbtTags.func_74764_b("result") || !nbtTags.func_74764_b("input"))
    	{
			Mekanism.logger.error("[Mekanism] Shaped recipe parse error: missing input or result compound tag.");
    		return null;
    	}
    	
    	ItemStack result = InventoryUtils.loadFromNBT(nbtTags.func_74775_l("result"));
    	NBTTagList list = nbtTags.func_150295_c("input", NBT.TAG_COMPOUND);
    	
    	if(result.func_190926_b() || list.func_74745_c() == 0)
    	{
    		Mekanism.logger.error("[Mekanism] Shaped recipe parse error: invalid result stack or input data list.");
    		return null;
    	}
    	
    	Object[] ret = new Object[list.func_74745_c()];
    	
    	for(int i = 0; i < list.func_74745_c(); i++)
    	{
    		NBTTagCompound compound = list.func_150305_b(i);
    		
    		if(compound.func_74764_b("oredict"))
    		{
    			ret[i] = compound.func_74779_i("oredict");
    		}
    		else if(compound.func_74764_b("pattern"))
    		{
    			ret[i] = compound.func_74779_i("pattern");
    		}
    		else if(compound.func_74764_b("character"))
    		{
    			String s = compound.func_74779_i("character");
    			
    			if(s.length() > 1)
    			{
    				Mekanism.logger.error("[Mekanism] Shaped recipe parse error: invalid pattern character data.");
    				return null;
    			}
    			
    			ret[i] = compound.func_74779_i("character").toCharArray()[0];
    		}
    		else if(compound.func_74764_b("itemstack"))
    		{
    			ret[i] = InventoryUtils.loadFromNBT(compound.func_74775_l("itemstack"));
    		}
    		else {
    			Mekanism.logger.error("[Mekanism] Shaped recipe parse error: invalid input tag data key.");
    			return null;
    		}
    	}
    	
    	return new ShapedMekanismRecipe(result, ret);
	}
}
