package mekanism.common.item;

import java.util.ArrayList;
import java.util.List;

import mekanism.api.Coord4D;
import mekanism.api.EnumColor;
import mekanism.api.Range4D;
import mekanism.api.energy.IEnergizedItem;
import mekanism.api.energy.IStrictEnergyStorage;
import mekanism.client.MekKeyHandler;
import mekanism.client.MekanismKeyHandler;
import mekanism.common.Mekanism;
import mekanism.common.MekanismBlocks;
import mekanism.common.Tier.BaseTier;
import mekanism.common.Tier.BinTier;
import mekanism.common.Tier.InductionCellTier;
import mekanism.common.Tier.InductionProviderTier;
import mekanism.common.base.ITierItem;
import mekanism.common.block.states.BlockStateBasic.BasicBlockType;
import mekanism.common.inventory.InventoryBin;
import mekanism.common.network.PacketTileEntity.TileEntityMessage;
import mekanism.common.tile.TileEntityBin;
import mekanism.common.tile.TileEntityInductionCell;
import mekanism.common.tile.TileEntityInductionProvider;
import mekanism.common.tile.TileEntityMultiblock;
import mekanism.common.util.ItemDataUtils;
import mekanism.common.util.LangUtils;
import mekanism.common.util.MekanismUtils;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.settings.GameSettings;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemBlock;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

/**
 * Item class for handling multiple metal block IDs.
 * 0:0: Osmium Block
 * 0:1: Bronze Block
 * 0:2: Refined Obsidian
 * 0:3: Charcoal Block
 * 0:4: Refined Glowstone
 * 0:5: Steel Block
 * 0:6: Bin
 * 0:7: Teleporter Frame
 * 0:8: Steel Casing
 * 0:9: Dynamic Tank
 * 0:10: Structural Glass
 * 0:11: Dynamic Valve
 * 0:12: Copper Block
 * 0:13: Tin Block
 * 0:14: Thermal Evaporation Controller
 * 0:15: Thermal Evaporation Valve
 * 1:0: Thermal Evaporation Block
 * 1:1: Induction Casing
 * 1:2: Induction Port
 * 1:3: Induction Cell
 * 1:4: Induction Provider
 * 1:5: Superheating Element
 * 1:6: Pressure Disperser
 * 1:7: Boiler Casing
 * 1:8: Boiler Valve
 * 1:9: Security Desk
 * @author AidanBrady
 *
 */
public class ItemBlockBasic extends ItemBlock implements IEnergizedItem, ITierItem
{
	public Block metaBlock;

	public ItemBlockBasic(Block block)
	{
		super(block);
		metaBlock = block;
		func_77627_a(true);
	}
	
	@Override
	public int getItemStackLimit(ItemStack stack)
    {
		if(BasicBlockType.get(stack) == BasicBlockType.BIN)
		{
			return new InventoryBin(stack).getItemCount() == 0 ? super.getItemStackLimit(stack) : 1;
		}
		
		return super.getItemStackLimit(stack);
    }
	
	public ItemStack getUnchargedCell(InductionCellTier tier)
	{
		ItemStack stack = new ItemStack(MekanismBlocks.BasicBlock2, 1, 3);
		setBaseTier(stack, tier.getBaseTier());
		
		return stack;
	}
	
	public ItemStack getUnchargedProvider(InductionProviderTier tier)
	{
		ItemStack stack = new ItemStack(MekanismBlocks.BasicBlock2, 1, 4);
		setBaseTier(stack, tier.getBaseTier());
		
		return stack;
	}
	
	@Override
	public BaseTier getBaseTier(ItemStack itemstack)
	{
		if(itemstack.func_77978_p() == null)
		{
			return BaseTier.BASIC;
		}

		return BaseTier.values()[itemstack.func_77978_p().func_74762_e("tier")];
	}

	@Override
	public void setBaseTier(ItemStack itemstack, BaseTier tier)
	{
		if(itemstack.func_77978_p() == null)
		{
			itemstack.func_77982_d(new NBTTagCompound());
		}

		itemstack.func_77978_p().func_74768_a("tier", tier.ordinal());
	}

	@Override
	public int func_77647_b(int i)
	{
		return i;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_77624_a(ItemStack itemstack, EntityPlayer entityplayer, List<String> list, boolean flag)
	{
		BasicBlockType type = BasicBlockType.get(itemstack);
		
		if(type.hasDescription)
		{
			if(!MekKeyHandler.getIsKeyPressed(MekanismKeyHandler.sneakKey))
			{
				if(type == BasicBlockType.BIN)
				{
					InventoryBin inv = new InventoryBin(itemstack);
		
					if(inv.getItemCount() > 0)
					{
						list.add(EnumColor.BRIGHT_GREEN + inv.getItemType().func_82833_r());
						String amountStr = inv.getItemCount() == Integer.MAX_VALUE ? LangUtils.localize("gui.infinite") : "" + inv.getItemCount();
						list.add(EnumColor.PURPLE + LangUtils.localize("tooltip.itemAmount") + ": " + EnumColor.GREY + amountStr);
					}
					else {
						list.add(EnumColor.DARK_RED + LangUtils.localize("gui.empty"));
					}
					
					int cap = BinTier.values()[getBaseTier(itemstack).ordinal()].storage;
					list.add(EnumColor.INDIGO + LangUtils.localize("tooltip.capacity") + ": " + EnumColor.GREY + (cap == Integer.MAX_VALUE ? LangUtils.localize("gui.infinite") : cap) + " " + LangUtils.localize("transmission.Items"));
				}
				else if(type == BasicBlockType.INDUCTION_CELL)
				{
					InductionCellTier tier = InductionCellTier.values()[getBaseTier(itemstack).ordinal()];
					
					list.add(tier.getBaseTier().getColor() + LangUtils.localize("tooltip.capacity") + ": " + EnumColor.GREY + MekanismUtils.getEnergyDisplay(tier.maxEnergy));
				}
				else if(type == BasicBlockType.INDUCTION_PROVIDER)
				{
					InductionProviderTier tier = InductionProviderTier.values()[getBaseTier(itemstack).ordinal()];
					
					list.add(tier.getBaseTier().getColor() + LangUtils.localize("tooltip.outputRate") + ": " + EnumColor.GREY + MekanismUtils.getEnergyDisplay(tier.output));
				}
				
				if(getMaxEnergy(itemstack) > 0)
				{
					list.add(EnumColor.BRIGHT_GREEN + LangUtils.localize("tooltip.storedEnergy") + ": " + EnumColor.GREY + MekanismUtils.getEnergyDisplay(getEnergy(itemstack)));
				}
				
				list.add(LangUtils.localize("tooltip.hold") + " " + EnumColor.INDIGO + GameSettings.func_74298_c(MekanismKeyHandler.sneakKey.func_151463_i()) + EnumColor.GREY + " " + LangUtils.localize("tooltip.forDetails") + ".");
			}
			else {
				list.addAll(MekanismUtils.splitTooltip(type.getDescription(), itemstack));
			}
		}
	}

	@Override
	public boolean hasContainerItem(ItemStack stack)
	{
		return BasicBlockType.get(stack) == BasicBlockType.BIN && ItemDataUtils.hasData(stack, "newCount");
	}

	@Override
	public ItemStack getContainerItem(ItemStack stack)
	{
		if(BasicBlockType.get(stack) == BasicBlockType.BIN)
		{
			if(!ItemDataUtils.hasData(stack, "newCount"))
			{
				return ItemStack.field_190927_a;
			}
			
			int newCount = ItemDataUtils.getInt(stack, "newCount");
			ItemDataUtils.removeData(stack, "newCount");

            ItemStack ret = stack.func_77946_l();
            ItemDataUtils.setInt(ret, "itemCount", newCount);

            return ret;
		}

		return ItemStack.field_190927_a;
	}

	@Override
	public boolean placeBlockAt(ItemStack stack, EntityPlayer player, World world, BlockPos pos, EnumFacing side, float hitX, float hitY, float hitZ, IBlockState state)
	{
		boolean place = true;
		
		BasicBlockType type = BasicBlockType.get(stack);
		
		if(type == BasicBlockType.SECURITY_DESK)
		{
			if(pos.func_177956_o()+1 > 255 || !world.func_180495_p(pos.func_177984_a()).func_177230_c().func_176200_f(world, pos.func_177984_a()))
			{
				place = false;
			}
		}

		if(place && super.placeBlockAt(stack, player, world, pos, side, hitX, hitY, hitZ, state))
		{
			if(type == BasicBlockType.BIN && stack.func_77978_p() != null)
			{
				TileEntityBin tileEntity = (TileEntityBin)world.func_175625_s(pos);
				InventoryBin inv = new InventoryBin(stack);
				
				tileEntity.tier = BinTier.values()[getBaseTier(stack).ordinal()];

				if(!inv.getItemType().func_190926_b())
				{
					tileEntity.setItemType(inv.getItemType());
				}

				tileEntity.setItemCount(inv.getItemCount());
			}
			else if(type == BasicBlockType.INDUCTION_CELL)
			{
				TileEntityInductionCell tileEntity = (TileEntityInductionCell)world.func_175625_s(pos);
				tileEntity.tier = InductionCellTier.values()[getBaseTier(stack).ordinal()];

				if(!world.field_72995_K)
				{
					Mekanism.packetHandler.sendToReceivers(new TileEntityMessage(Coord4D.get(tileEntity), tileEntity.getNetworkedData(new ArrayList<Object>())), new Range4D(Coord4D.get(tileEntity)));
				}
			}
			else if(type == BasicBlockType.INDUCTION_PROVIDER)
			{
				TileEntityInductionProvider tileEntity = (TileEntityInductionProvider)world.func_175625_s(pos);
				tileEntity.tier = InductionProviderTier.values()[getBaseTier(stack).ordinal()];

				if(!world.field_72995_K)
				{
					Mekanism.packetHandler.sendToReceivers(new TileEntityMessage(Coord4D.get(tileEntity), tileEntity.getNetworkedData(new ArrayList<Object>())), new Range4D(Coord4D.get(tileEntity)));
				}
			}

			TileEntity tileEntity = world.func_175625_s(pos);

			if(tileEntity instanceof IStrictEnergyStorage && !(tileEntity instanceof TileEntityMultiblock<?>))
			{
				((IStrictEnergyStorage)tileEntity).setEnergy(getEnergy(stack));
			}
		}

		return place;
	}
	
	@Override
	public String func_77667_c(ItemStack itemstack)
	{
		BasicBlockType type = BasicBlockType.get(itemstack);
		
		if(type != null)
		{
			String name = func_77658_a() + "." + BasicBlockType.get(itemstack).name;
			
			if(type == BasicBlockType.BIN || type == BasicBlockType.INDUCTION_CELL || type == BasicBlockType.INDUCTION_PROVIDER)
			{
				name += getBaseTier(itemstack).getSimpleName();
			}
			
			return name;
		}

		return "null";
	}
	
	@Override
	public double getEnergy(ItemStack itemStack)
	{
		if(BasicBlockType.get(itemStack) == BasicBlockType.INDUCTION_CELL)
		{
			return ItemDataUtils.getDouble(itemStack, "energyStored");
		}
		
		return 0;
	}

	@Override
	public void setEnergy(ItemStack itemStack, double amount)
	{
		if(BasicBlockType.get(itemStack) == BasicBlockType.INDUCTION_CELL)
		{
			ItemDataUtils.setDouble(itemStack, "energyStored", Math.max(Math.min(amount, getMaxEnergy(itemStack)), 0));
		}
	}

	@Override
	public double getMaxEnergy(ItemStack itemStack)
	{
		if(BasicBlockType.get(itemStack) == BasicBlockType.INDUCTION_CELL)
		{
			return InductionCellTier.values()[getBaseTier(itemStack).ordinal()].maxEnergy;
		}
		
		return 0;
	}

	@Override
	public double getMaxTransfer(ItemStack itemStack)
	{
		return 0;
	}

	@Override
	public boolean canReceive(ItemStack itemStack)
	{
		return false;
	}

	@Override
	public boolean canSend(ItemStack itemStack)
	{
		return false;
	}
}
