package mekanism.common.frequency;

import io.netty.buffer.ByteBuf;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import mekanism.api.Coord4D;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.world.World;
import net.minecraft.world.WorldSavedData;
import net.minecraftforge.common.util.Constants.NBT;

public class FrequencyManager
{
	public static final int MAX_FREQ_LENGTH = 16;
	public static final List<Character> SPECIAL_CHARS = Arrays.asList('-', ' ', '|', '\'', '\"', '_', '+', ':', '(', ')', 
			'?', '!', '/', '@', '$', '`', '~', ',', '.', '#');

	public static boolean loaded;
	
	private static Set<FrequencyManager> managers = new HashSet<FrequencyManager>();
	
	private Set<Frequency> frequencies = new HashSet<Frequency>();
	
	private FrequencyDataHandler dataHandler;
	
	private UUID ownerUUID;
	
	private String name;
	
	private Class<? extends Frequency> frequencyClass;
	
	public FrequencyManager(Class c, String n)
	{
		frequencyClass = c;
		name = n;
		
		managers.add(this);
	}
	
	public FrequencyManager(Class c, String n, UUID uuid)
	{
		this(c, n);
		
		ownerUUID = uuid;
	}
	
	public static void load(World world)
	{
		loaded = true;
		
		for(FrequencyManager manager : managers)
		{
			manager.createOrLoad(world);
		}
	}
	
	public Frequency update(Coord4D coord, Frequency freq)
	{
		for(Frequency iterFreq : frequencies)
		{
			if(freq.equals(iterFreq))
			{
				iterFreq.activeCoords.add(coord);
				dataHandler.func_76185_a();
				
				return iterFreq;
			}
		}
		
		deactivate(coord);
		
		return null;
	}
	
	public void remove(String name, UUID owner)
	{
		for(Iterator<Frequency> iter = getFrequencies().iterator(); iter.hasNext();)
		{
			Frequency iterFreq = iter.next();
			
			if(iterFreq.name.equals(name) && iterFreq.ownerUUID.equals(owner))
			{
				iter.remove();
				dataHandler.func_76185_a();
			}
		}
	}
	
	public void remove(String name)
	{
		for(Iterator<Frequency> iter = getFrequencies().iterator(); iter.hasNext();)
		{
			Frequency iterFreq = iter.next();
			
			if(iterFreq.name.equals(name))
			{
				iter.remove();
				dataHandler.func_76185_a();
			}
		}
	}
	
	public int removeAll(String user)
	{
		int amount = 0;
		
		for(Iterator<Frequency> iter = getFrequencies().iterator(); iter.hasNext();)
		{
			Frequency iterFreq = iter.next();
			
			if(iterFreq.ownerUUID.equals(user))
			{
				iter.remove();
				dataHandler.func_76185_a();
				amount++;
			}
		}
		
		return amount;
	}
	
	public void deactivate(Coord4D coord)
	{
		for(Frequency freq : frequencies)
		{
			freq.activeCoords.remove(coord);
			dataHandler.func_76185_a();
		}
	}
	
	public Frequency validateFrequency(UUID uuid, Coord4D coord, Frequency freq)
	{
		for(Frequency iterFreq : frequencies)
		{
			if(freq.equals(iterFreq))
			{
				iterFreq.activeCoords.add(coord);
				dataHandler.func_76185_a();
				
				return iterFreq;
			}
		}
		
		if(uuid.equals(freq.ownerUUID))
		{
			freq.activeCoords.add(coord);
			freq.valid = true;
			frequencies.add(freq);
			dataHandler.func_76185_a();
			
			return freq;
		}
		
		return null;
	}
	
	public void createOrLoad(World world)
	{
		String name = getName();
		
		if(dataHandler == null)
		{
			dataHandler = (FrequencyDataHandler)world.getPerWorldStorage().func_75742_a(FrequencyDataHandler.class, name);
			
			if(dataHandler == null)
			{
				dataHandler = new FrequencyDataHandler(name);
				dataHandler.setManager(this);
				world.getPerWorldStorage().func_75745_a(name, dataHandler);
			}
			else {
				dataHandler.setManager(this);
				dataHandler.syncManager();
			}
		}
	}
	
	public Set<Frequency> getFrequencies()
	{
		return frequencies;
	}
	
	public void addFrequency(Frequency freq)
	{
		frequencies.add(freq);
		dataHandler.func_76185_a();
	}
	
	public boolean containsFrequency(String name)
	{
		for(Frequency freq : frequencies)
		{
			if(freq.name.equals(name))
			{
				return true;
			}
		}
		
		return false;
	}
	
	public static void tick(World world)
	{
		if(!loaded)
		{
			load(world);
		}
		
		for(FrequencyManager manager : managers)
		{
			manager.tickSelf(world);
		}
	}

	public void tickSelf(World world)
	{
		for(Frequency iterFreq : frequencies)
		{
			for(Iterator<Coord4D> iter = iterFreq.activeCoords.iterator(); iter.hasNext();)
			{
				Coord4D coord = iter.next();
				
				if(coord.dimensionId == world.field_73011_w.getDimension())
				{
					if(!coord.exists(world))
					{
						iter.remove();
					}
					else {
						TileEntity tile = coord.getTileEntity(world);
						
						if(!(tile instanceof IFrequencyHandler))
						{
							iter.remove();
						}
						else {
							Frequency freq = ((IFrequencyHandler)tile).getFrequency(this);
							
							if(freq == null || !freq.equals(iterFreq))
							{
								iter.remove();
							}
						}
					}
				}
			}
		}
	}
	
	public void writeFrequencies(ArrayList data)
	{
		data.add(frequencies.size());
		
		for(Frequency freq : frequencies)
		{
			freq.write(data);
		}
	}
	
	public Set<Frequency> readFrequencies(ByteBuf dataStream)
	{
		Set<Frequency> ret = new HashSet<Frequency>();
		int size = dataStream.readInt();
		
		try {
			for(int i = 0; i < size; i++)
			{
				Frequency freq = frequencyClass.getConstructor(new Class[] {ByteBuf.class}).newInstance(dataStream);
				freq.read(dataStream);
				ret.add(freq);
			}
		} catch(Exception e) {
			e.printStackTrace();
		}
		
		return ret;
	}
	
	public String getName()
	{
		return ownerUUID != null ? (ownerUUID.toString() + "_" + name + "FrequencyHandler") : (name + "FrequencyHandler");
	}
	
	public static void reset()
	{
		for(FrequencyManager manager : managers)
		{
			manager.frequencies.clear();
			manager.dataHandler = null;
		}
		
		loaded = false;
	}
	
	public static class FrequencyDataHandler extends WorldSavedData
	{
		public FrequencyManager manager;
		
		public Set<Frequency> loadedFrequencies;
		public UUID loadedOwner;
		
		public FrequencyDataHandler(String tagName)
		{
			super(tagName);
		}
		
		public void setManager(FrequencyManager m)
		{
			manager = m;
		}
		
		public void syncManager()
		{
			if(loadedFrequencies != null)
			{
				manager.frequencies = loadedFrequencies;
				manager.ownerUUID = loadedOwner;
			}
		}
		
		@Override
		public void func_76184_a(NBTTagCompound nbtTags) 
		{
			try {
				String frequencyClass = nbtTags.func_74779_i("frequencyClass");
				
				if(nbtTags.func_74764_b("ownerUUID"))
				{
					loadedOwner = UUID.fromString(nbtTags.func_74779_i("ownerUUID"));
				}
				
				NBTTagList list = nbtTags.func_150295_c("freqList", NBT.TAG_COMPOUND);
				
				loadedFrequencies = new HashSet<Frequency>();
				
				for(int i = 0; i < list.func_74745_c(); i++)
				{
					NBTTagCompound compound = list.func_150305_b(i);
					
					Constructor c = Class.forName(frequencyClass).getConstructor(new Class[] {NBTTagCompound.class});
					Frequency freq = (Frequency)c.newInstance(compound);
					
					loadedFrequencies.add(freq);
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}

		@Override
		public NBTTagCompound func_189551_b(NBTTagCompound nbtTags) 
		{
			nbtTags.func_74778_a("frequencyClass", manager.frequencyClass.getName());
			
			if(manager.ownerUUID != null)
			{
				nbtTags.func_74778_a("ownerUUID", manager.ownerUUID.toString());
			}
			
			NBTTagList list = new NBTTagList();
			
			for(Frequency freq : manager.getFrequencies())
			{
				NBTTagCompound compound = new NBTTagCompound();
				freq.write(compound);
				list.func_74742_a(compound);
			}
			
			nbtTags.func_74782_a("freqList", list);
			
			return nbtTags;
		}
	}
}
