package mekanism.common.block;

import java.util.Random;
import java.util.UUID;

import mekanism.api.Coord4D;
import mekanism.api.energy.IEnergizedItem;
import mekanism.api.energy.IStrictEnergyStorage;
import mekanism.common.Mekanism;
import mekanism.common.Tier.FluidTankTier;
import mekanism.common.base.IActiveState;
import mekanism.common.base.IBoundingBlock;
import mekanism.common.base.IFactory;
import mekanism.common.base.IFactory.RecipeType;
import mekanism.common.base.IRedstoneControl;
import mekanism.common.base.ISideConfiguration;
import mekanism.common.base.ISustainedData;
import mekanism.common.base.ISustainedInventory;
import mekanism.common.base.ISustainedTank;
import mekanism.common.base.ITierItem;
import mekanism.common.base.IUpgradeTile;
import mekanism.common.block.states.BlockStateFacing;
import mekanism.common.block.states.BlockStateMachine;
import mekanism.common.block.states.BlockStateMachine.MachineBlock;
import mekanism.common.block.states.BlockStateMachine.MachineType;
import mekanism.common.config.MekanismConfig.client;
import mekanism.common.content.entangloporter.InventoryFrequency;
import mekanism.common.item.ItemBlockMachine;
import mekanism.common.network.PacketLogisticalSorterGui.LogisticalSorterGuiMessage;
import mekanism.common.network.PacketLogisticalSorterGui.SorterGuiPacket;
import mekanism.common.security.ISecurityItem;
import mekanism.common.security.ISecurityTile;
import mekanism.common.tile.TileEntityFactory;
import mekanism.common.tile.TileEntityFluidTank;
import mekanism.common.tile.TileEntityLaser;
import mekanism.common.tile.TileEntityLaserAmplifier;
import mekanism.common.tile.TileEntityLogisticalSorter;
import mekanism.common.tile.TileEntityQuantumEntangloporter;
import mekanism.common.tile.TileEntityMetallurgicInfuser;
import mekanism.common.tile.TileEntityPersonalChest;
import mekanism.common.tile.prefab.TileEntityBasicBlock;
import mekanism.common.tile.prefab.TileEntityContainerBlock;
import mekanism.common.util.FluidContainerUtils;
import mekanism.common.util.ItemDataUtils;
import mekanism.common.util.MekanismUtils;
import mekanism.common.util.MultipartUtils;
import mekanism.common.util.PipeUtils;
import mekanism.common.util.SecurityUtils;
import mekanism.common.util.StackUtils;
import net.minecraft.block.Block;
import net.minecraft.block.BlockContainer;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.Explosion;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidUtil;
import net.minecraftforge.fluids.capability.IFluidHandlerItem;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import buildcraft.api.tools.IToolWrench;

/**
 * Block class for handling multiple machine block IDs.
 * 0:0: Enrichment Chamber
 * 0:1: Osmium Compressor
 * 0:2: Combiner
 * 0:3: Crusher
 * 0:4: Digital Miner
 * 0:5: Basic Factory
 * 0:6: Advanced Factory
 * 0:7: Elite Factory
 * 0:8: Metallurgic Infuser
 * 0:9: Purification Chamber
 * 0:10: Energized Smelter
 * 0:11: Teleporter
 * 0:12: Electric Pump
 * 0:13: Electric Chest
 * 0:14: Chargepad
 * 0:15: Logistical Sorter
 * 1:0: Rotary Condensentrator
 * 1:1: Chemical Oxidizer
 * 1:2: Chemical Infuser
 * 1:3: Chemical Injection Chamber
 * 1:4: Electrolytic Separator
 * 1:5: Precision Sawmill
 * 1:6: Chemical Dissolution Chamber
 * 1:7: Chemical Washer
 * 1:8: Chemical Crystallizer
 * 1:9: Seismic Vibrator
 * 1:10: Pressurized Reaction Chamber
 * 1:11: Fluid Tank
 * 1:12: Fluidic Plenisher
 * 1:13: Laser
 * 1:14: Laser Amplifier
 * 1:15: Laser Tractor Beam
 * 2:0: Quantum Entangloporter
 * 2:1: Solar Neutron Activator
 * 2:2: Ambient Accumulator
 * 2:3: Oredictionificator
 * 2:4: Resistive Heater
 * 2:5: Formulaic Assemblicator
 * 2:6: Fuelwood Heater
 * 
 * @author AidanBrady
 *
 */
public abstract class BlockMachine extends BlockContainer
{
	private static final AxisAlignedBB CHARGEPAD_BOUNDS = new AxisAlignedBB(0.0F, 0.0F, 0.0F, 1.0F, 0.06F, 1.0F);
	private static final AxisAlignedBB TANK_BOUNDS = new AxisAlignedBB(0.125F, 0.0F, 0.125F, 0.875F, 1.0F, 0.875F);
	private static final AxisAlignedBB LASER_BOUNDS = new AxisAlignedBB(0.25F, 0.0F, 0.25F, 0.75F, 1.0F, 0.75F);
	private static final AxisAlignedBB LOGISTICAL_SORTER_BOUNDS = new AxisAlignedBB(0.125F, 0.0F, 0.125F, 0.875F, 1.0F, 0.875F);

	public BlockMachine()
	{
		super(Material.field_151573_f);
		func_149711_c(3.5F);
		func_149752_b(16F);
		func_149647_a(Mekanism.tabMekanism);
	}

	public static BlockMachine getBlockMachine(MachineBlock block)
	{
		return new BlockMachine()
		{
			@Override
			public MachineBlock getMachineBlock()
			{
				return block;
			}
		};
	}

	public abstract MachineBlock getMachineBlock();

	@Override
	public BlockStateContainer func_180661_e()
	{
		return new BlockStateMachine(this, getTypeProperty());
	}

	@Override
	public IBlockState func_176203_a(int meta)
	{
		MachineType type = MachineType.get(getMachineBlock(), meta & 0xF);

		return func_176223_P().func_177226_a(getTypeProperty(), type);
	}

	@Override
	public int func_176201_c(IBlockState state)
	{
		MachineType type = state.func_177229_b(getTypeProperty());
		return type.meta;
	}

	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess worldIn, BlockPos pos)
	{
		TileEntity tile = MekanismUtils.getTileEntitySafe(worldIn, pos);
		
		if(tile instanceof TileEntityBasicBlock && ((TileEntityBasicBlock)tile).facing != null)
		{
			state = state.func_177226_a(BlockStateFacing.facingProperty, ((TileEntityBasicBlock)tile).facing);
		}
		
		if(tile instanceof IActiveState)
		{
			state = state.func_177226_a(BlockStateMachine.activeProperty, ((IActiveState)tile).getActive());
		}
		
		if(tile instanceof TileEntityFluidTank)
		{
			state = state.func_177226_a(BlockStateMachine.tierProperty, ((TileEntityFluidTank)tile).tier.getBaseTier());
		}
		
		if(tile instanceof TileEntityFactory)
		{
			state = state.func_177226_a(BlockStateMachine.recipeProperty, ((TileEntityFactory)tile).recipeType);
		}
		
		return state;
	}

	@Override
	public void func_180633_a(World world, BlockPos pos, IBlockState state, EntityLivingBase placer, ItemStack stack)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		int side = MathHelper.func_76128_c((placer.field_70177_z * 4.0F / 360.0F) + 0.5D) & 3;
		int height = Math.round(placer.field_70125_A);
		int change = 3;

		if(tileEntity == null)
		{
			return;
		}

		if(tileEntity.canSetFacing(0) && tileEntity.canSetFacing(1))
		{
			if(height >= 65)
			{
				change = 1;
			} 
			else if(height <= -65)
			{
				change = 0;
			}
		}

		if(change != 0 && change != 1)
		{
			switch(side)
			{
				case 0:
					change = 2;
					break;
				case 1:
					change = 5;
					break;
				case 2:
					change = 3;
					break;
				case 3:
					change = 4;
					break;
			}
		}

		tileEntity.setFacing((short)change);
		tileEntity.redstone = world.func_175687_A(pos) > 0;

		if(tileEntity instanceof TileEntityLogisticalSorter)
		{
			TileEntityLogisticalSorter transporter = (TileEntityLogisticalSorter)tileEntity;

			if(!transporter.hasInventory())
			{
				for(EnumFacing dir : EnumFacing.field_82609_l)
				{
					TileEntity tile = Coord4D.get(transporter).offset(dir).getTileEntity(world);

					if(tile instanceof IInventory)
					{
						tileEntity.setFacing((short)dir.func_176734_d().ordinal());
						break;
					}
				}
			}
		}

		if(tileEntity instanceof IBoundingBlock)
		{
			((IBoundingBlock)tileEntity).onPlace();
		}
	}

	@Override
	public void func_180663_b(World world, BlockPos pos, IBlockState state)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);

		if(tileEntity instanceof IBoundingBlock)
		{
			((IBoundingBlock)tileEntity).onBreak();
		}

		super.func_180663_b(world, pos, state);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_180655_c(IBlockState state, World world, BlockPos pos, Random random)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		
		if(tileEntity instanceof TileEntityFluidTank)
		{
			return;
		}

		if(MekanismUtils.isActive(world, pos) && ((IActiveState)tileEntity).renderUpdate() && client.machineEffects)
		{
			float xRandom = (float)pos.func_177958_n() + 0.5F;
			float yRandom = (float)pos.func_177956_o() + 0.0F + random.nextFloat() * 6.0F / 16.0F;
			float zRandom = (float)pos.func_177952_p() + 0.5F;
			float iRandom = 0.52F;
			float jRandom = random.nextFloat() * 0.6F - 0.3F;

			EnumFacing side = tileEntity.facing;

			if(tileEntity instanceof TileEntityMetallurgicInfuser)
			{
				side = side.func_176734_d();
			}

			switch(side)
			{
				case WEST:
					world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (xRandom - iRandom), yRandom, (zRandom + jRandom), 0.0D, 0.0D, 0.0D);
					world.func_175688_a(EnumParticleTypes.REDSTONE, (xRandom - iRandom), yRandom, (zRandom + jRandom), 0.0D, 0.0D, 0.0D);
					break;
				case EAST:
					world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (xRandom + iRandom), yRandom, (zRandom + jRandom), 0.0D, 0.0D, 0.0D);
					world.func_175688_a(EnumParticleTypes.REDSTONE, (xRandom + iRandom), yRandom, (zRandom + jRandom), 0.0D, 0.0D, 0.0D);
					break;
				case NORTH:
					world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (xRandom + jRandom), yRandom, (zRandom - iRandom), 0.0D, 0.0D, 0.0D);
					world.func_175688_a(EnumParticleTypes.REDSTONE, (xRandom + jRandom), yRandom, (zRandom - iRandom), 0.0D, 0.0D, 0.0D);
					break;
				case SOUTH:
					world.func_175688_a(EnumParticleTypes.SMOKE_NORMAL, (xRandom + jRandom), yRandom, (zRandom + iRandom), 0.0D, 0.0D, 0.0D);
					world.func_175688_a(EnumParticleTypes.REDSTONE, (xRandom + jRandom), yRandom, (zRandom + iRandom), 0.0D, 0.0D, 0.0D);
					break;
				default:
					break;
			}
		}
	}

	@Override
	public int getLightValue(IBlockState state, IBlockAccess world, BlockPos pos)
	{
		if(client.enableAmbientLighting)
		{
			TileEntity tileEntity = MekanismUtils.getTileEntitySafe(world, pos);

			if(tileEntity instanceof IActiveState)
			{
				if(((IActiveState)tileEntity).getActive() && ((IActiveState)tileEntity).lightUpdate())
				{
					return client.ambientLightingLevel;
				}
			}
		}

		return 0;
	}

	@Override
	public int func_180651_a(IBlockState state)
	{
		return state.func_177230_c().func_176201_c(state);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_149666_a(Item item, CreativeTabs creativetabs, NonNullList<ItemStack> list)
	{
		for(MachineType type : MachineType.getValidMachines())
		{
			if(type.typeBlock == getMachineBlock() && type.isEnabled())
			{
				switch(type)
				{
					case BASIC_FACTORY:
					case ADVANCED_FACTORY:
					case ELITE_FACTORY:
						for(RecipeType recipe : RecipeType.values())
						{
							ItemStack stack = new ItemStack(item, 1, type.meta);
							((IFactory)stack.func_77973_b()).setRecipeType(recipe.ordinal(), stack);
							list.add(stack);
						}

						break;
					case FLUID_TANK:
						ItemBlockMachine itemMachine = (ItemBlockMachine)item;

						for(FluidTankTier tier : FluidTankTier.values())
						{
							ItemStack stack = new ItemStack(item, 1, type.meta);
							itemMachine.setBaseTier(stack, tier.getBaseTier());
							list.add(stack);
						}

						break;
					default:
						list.add(new ItemStack(item, 1, type.meta));
				}
			}
		}
	}

	@Override
	public boolean func_180639_a(World world, BlockPos pos, IBlockState state, EntityPlayer entityplayer, EnumHand hand, EnumFacing side, float hitX, float hitY, float hitZ)
	{
		if(world.field_72995_K)
		{
			return true;
		}

		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		int metadata = state.func_177230_c().func_176201_c(state);
		ItemStack stack = entityplayer.func_184586_b(hand);

		if(!stack.func_190926_b())
		{
			Item tool = stack.func_77973_b();

			if(MekanismUtils.hasUsableWrench(entityplayer, pos))
			{
				if(SecurityUtils.canAccess(entityplayer, tileEntity))
				{
					if(entityplayer.func_70093_af())
					{
						dismantleBlock(state, world, pos, false);

						return true;
					}

					if(MekanismUtils.isBCWrench(tool))
					{
						((IToolWrench)tool).wrenchUsed(entityplayer, hand, stack, new RayTraceResult(new Vec3d(hitX, hitY, hitZ), side, pos));
					}

					int change = tileEntity.facing.func_176746_e().ordinal();

					if(tileEntity instanceof TileEntityLogisticalSorter)
					{
						if(!((TileEntityLogisticalSorter)tileEntity).hasInventory())
						{
							for(EnumFacing dir : EnumFacing.field_82609_l)
							{
								TileEntity tile = Coord4D.get(tileEntity).offset(dir).getTileEntity(world);

								if(tile instanceof IInventory)
								{
									change = dir.func_176734_d().ordinal();
									break;
								}
							}
						}
					}

					tileEntity.setFacing((short)change);
					world.func_175685_c(pos, this, true);
				} 
				else {
					SecurityUtils.displayNoAccess(entityplayer);
				}

				return true;
			}
		}

		if(tileEntity != null)
		{
			MachineType type = MachineType.get(getMachineBlock(), metadata);

			switch(type)
			{
				case PERSONAL_CHEST:
					if(!entityplayer.func_70093_af() && !world.isSideSolid(pos.func_177984_a(), EnumFacing.DOWN))
					{
						TileEntityPersonalChest chest = (TileEntityPersonalChest)tileEntity;

						if(SecurityUtils.canAccess(entityplayer, tileEntity))
						{
							MekanismUtils.openPersonalChestGui((EntityPlayerMP)entityplayer, chest, null, true);
						} 
						else {
							SecurityUtils.displayNoAccess(entityplayer);
						}

						return true;
					}

					break;
				case FLUID_TANK:
					if(!entityplayer.func_70093_af())
					{
						if(SecurityUtils.canAccess(entityplayer, tileEntity))
						{
							if(!stack.func_190926_b() && FluidContainerUtils.isFluidContainer(stack))
							{
								if(manageInventory(entityplayer, (TileEntityFluidTank)tileEntity, hand, stack))
								{
									entityplayer.field_71071_by.func_70296_d();
									return true;
								}
							} 
							else {
								entityplayer.openGui(Mekanism.instance, type.guiId, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
							}
						} 
						else {
							SecurityUtils.displayNoAccess(entityplayer);
						}

						return true;
					}

					break;
				case LOGISTICAL_SORTER:
					if(!entityplayer.func_70093_af())
					{
						if(SecurityUtils.canAccess(entityplayer, tileEntity))
						{
							LogisticalSorterGuiMessage.openServerGui(SorterGuiPacket.SERVER, 0, world, (EntityPlayerMP)entityplayer, Coord4D.get(tileEntity), -1);
						} 
						else {
							SecurityUtils.displayNoAccess(entityplayer);
						}

						return true;
					}

					break;
				case TELEPORTER:
				case QUANTUM_ENTANGLOPORTER:
					if(!entityplayer.func_70093_af())
					{
						UUID owner = ((ISecurityTile)tileEntity).getSecurity().getOwnerUUID();
						
						if(MekanismUtils.isOp((EntityPlayerMP)entityplayer) || owner == null || entityplayer.func_110124_au().equals(owner))
						{
							entityplayer.openGui(Mekanism.instance, type.guiId, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
						} 
						else {
							SecurityUtils.displayNoAccess(entityplayer);
						}

						return true;
					}

					break;
				default:
					if(!entityplayer.func_70093_af() && type.guiId != -1)
					{
						if(SecurityUtils.canAccess(entityplayer, tileEntity))
						{
							entityplayer.openGui(Mekanism.instance, type.guiId, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
						} 
						else {
							SecurityUtils.displayNoAccess(entityplayer);
						}

						return true;
					}

					break;
			}
		}

		return false;
	}

	@Override
	public TileEntity createTileEntity(World world, IBlockState state)
	{
		int metadata = state.func_177230_c().func_176201_c(state);
		
		if(MachineType.get(getMachineBlock(), metadata) == null)
		{
			return null;
		}

		return MachineType.get(getMachineBlock(), metadata).create();
	}

	@Override
	public TileEntity func_149915_a(World world, int metadata)
	{
		return null;
	}

	@Override
	public boolean func_149662_c(IBlockState state)
	{
		return false;
	}

	@Override
	public Item func_180660_a(IBlockState state, Random random, int fortune)
	{
		return null;
	}

	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state)
	{
		return EnumBlockRenderType.MODEL;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public BlockRenderLayer func_180664_k()
	{
		return BlockRenderLayer.CUTOUT;
	}

	@Override
	public float func_180647_a(IBlockState state, EntityPlayer player, World world, BlockPos pos)
	{
		TileEntity tile = world.func_175625_s(pos);
		
		return SecurityUtils.canAccess(player, tile) ? super.func_180647_a(state, player, world, pos) : 0.0F;
	}
	
	@Override
	public float getExplosionResistance(World world, BlockPos pos, Entity exploder, Explosion explosion)
    {
		IBlockState state = world.func_180495_p(pos);
		if(MachineType.get(getMachineBlock(), state.func_177230_c().func_176201_c(state)) != MachineType.PERSONAL_CHEST)
		{
			return field_149781_w;
		}
		else {
			return -1;
		}
    }

	@Override
	public boolean removedByPlayer(IBlockState state, World world, BlockPos pos, EntityPlayer player, boolean willHarvest)
	{
		if(!player.field_71075_bZ.field_75098_d && !world.field_72995_K && willHarvest)
		{
			TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);

			float motion = 0.7F;
			double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

			EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, getPickBlock(state, null, world, pos, player));

			world.func_72838_d(entityItem);
		}

		return world.func_175698_g(pos);
	}

	@Override
	public boolean func_149740_M(IBlockState state)
	{
		return true;
	}
	
	@Override
	public int func_180641_l(IBlockState state, World world, BlockPos pos)
	{
		TileEntity tileEntity = world.func_175625_s(pos);
		
		if(tileEntity instanceof TileEntityFluidTank)
		{
			return ((TileEntityFluidTank)tileEntity).getRedstoneLevel();
		}
		
		if(tileEntity instanceof TileEntityLaserAmplifier)
		{
			TileEntityLaserAmplifier amplifier = (TileEntityLaserAmplifier)tileEntity;
			
			if(amplifier.outputMode == TileEntityLaserAmplifier.RedstoneOutput.ENERGY_CONTENTS)
			{
				return amplifier.getRedstoneLevel();
			}
			else {
				return func_180656_a(state, world, pos, null);
			}
		}
		
		return 0;
	}
	
	private boolean manageInventory(EntityPlayer player, TileEntityFluidTank tileEntity, EnumHand hand, ItemStack itemStack)
	{
		ItemStack copyStack = StackUtils.size(itemStack.func_77946_l(), 1);
		
		if(FluidContainerUtils.isFluidContainer(itemStack))
		{
			IFluidHandlerItem handler = FluidUtil.getFluidHandler(copyStack);
			
			if(FluidUtil.getFluidContained(copyStack) == null)
			{
				if(tileEntity.fluidTank.getFluid() != null)
				{
					int filled = handler.fill(tileEntity.fluidTank.getFluid(), !player.field_71075_bZ.field_75098_d);
					copyStack = handler.getContainer();
					
					if(filled > 0)
					{
						if(itemStack.func_190916_E() == 1)
						{
							player.func_184611_a(hand, copyStack);
						}
						else if(itemStack.func_190916_E() > 1 && player.field_71071_by.func_70441_a(copyStack))
						{
							itemStack.func_190918_g(1);
						}
						
						if(tileEntity.tier != FluidTankTier.CREATIVE)
						{
							tileEntity.fluidTank.drain(filled, true);
						}
						
						return true;
					}
				}
			}
			else {
				FluidStack itemFluid = FluidUtil.getFluidContained(copyStack);
				int needed = tileEntity.getCurrentNeeded();
				
				if(tileEntity.fluidTank.getFluid() != null && !tileEntity.fluidTank.getFluid().isFluidEqual(itemFluid))
				{
					return false;
				}
				
				boolean filled = false;
				FluidStack drained = handler.drain(needed, !player.field_71075_bZ.field_75098_d);
				copyStack = handler.getContainer();
				
				if(copyStack.func_190916_E() == 0)
				{
					copyStack = ItemStack.field_190927_a;
				}
				
				if(drained != null)
				{
					if(player.field_71075_bZ.field_75098_d)
					{
						filled = true;
					}
					else {
						if(!copyStack.func_190926_b())
						{
							if(itemStack.func_190916_E() == 1)
							{
								player.func_184611_a(hand, copyStack);
								filled = true;
							}
							else {
								if(player.field_71071_by.func_70441_a(copyStack))
								{
									itemStack.func_190918_g(1);
	
									filled = true;
								}
							}
						}
						else {
							itemStack.func_190918_g(1);
	
							if(itemStack.func_190916_E() == 0)
							{
								player.func_184611_a(hand, ItemStack.field_190927_a);
							}
	
							filled = true;
						}
					}
	
					if(filled)
					{
						int toFill = tileEntity.fluidTank.getCapacity()-tileEntity.fluidTank.getFluidAmount();
						
						if(tileEntity.tier != FluidTankTier.CREATIVE)
						{
							toFill = Math.min(toFill, drained.amount);
						}
						
						tileEntity.fluidTank.fill(PipeUtils.copy(drained, toFill), true);
						
						if(drained.amount-toFill > 0)
						{
							tileEntity.pushUp(PipeUtils.copy(itemFluid, drained.amount-toFill), true);
						}
						
						return true;
					}
				}
			}
		}

		return false;
	}

	@Override
	public void func_189540_a(IBlockState state, World world, BlockPos pos, Block neighborBlock, BlockPos neighborPos)
	{
		if(!world.field_72995_K)
		{
			TileEntity tileEntity = world.func_175625_s(pos);

			if(tileEntity instanceof TileEntityBasicBlock)
			{
				((TileEntityBasicBlock)tileEntity).onNeighborChange(neighborBlock);
			}
			
			if(tileEntity instanceof TileEntityLogisticalSorter)
			{
				TileEntityLogisticalSorter sorter = (TileEntityLogisticalSorter)tileEntity;

				if(!sorter.hasInventory())
				{
					for(EnumFacing dir : EnumFacing.field_82609_l)
					{
						TileEntity tile = Coord4D.get(tileEntity).offset(dir).getTileEntity(world);

						if(tile instanceof IInventory)
						{
							sorter.setFacing((short)dir.func_176734_d().ordinal());
							return;
						}
					}
				}
			}
		}
	}

	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player)
	{
		TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
		ItemStack itemStack = new ItemStack(this, 1, state.func_177230_c().func_176201_c(state));

		if(itemStack.func_77978_p() == null)
		{
			itemStack.func_77982_d(new NBTTagCompound());
		}
		
		if(tileEntity instanceof TileEntityFluidTank)
		{
			ITierItem tierItem = (ITierItem)itemStack.func_77973_b();
			tierItem.setBaseTier(itemStack, ((TileEntityFluidTank)tileEntity).tier.getBaseTier());
		}
		
		if(tileEntity instanceof ISecurityTile)
		{
			ISecurityItem securityItem = (ISecurityItem)itemStack.func_77973_b();
			
			if(securityItem.hasSecurity(itemStack))
			{
				securityItem.setOwnerUUID(itemStack, ((ISecurityTile)tileEntity).getSecurity().getOwnerUUID());
				securityItem.setSecurity(itemStack, ((ISecurityTile)tileEntity).getSecurity().getMode());
			}
		}

		if(tileEntity instanceof IUpgradeTile)
		{
			((IUpgradeTile)tileEntity).getComponent().write(ItemDataUtils.getDataMap(itemStack));
		}

		if(tileEntity instanceof ISideConfiguration)
		{
			ISideConfiguration config = (ISideConfiguration)tileEntity;

			config.getConfig().write(ItemDataUtils.getDataMap(itemStack));
			config.getEjector().write(ItemDataUtils.getDataMap(itemStack));
		}
		
		if(tileEntity instanceof ISustainedData)
		{
			((ISustainedData)tileEntity).writeSustainedData(itemStack);
		}

		if(tileEntity instanceof IRedstoneControl)
		{
			IRedstoneControl control = (IRedstoneControl)tileEntity;
			ItemDataUtils.setInt(itemStack, "controlType", control.getControlType().ordinal());
		}

		if(tileEntity instanceof IStrictEnergyStorage)
		{
			IEnergizedItem energizedItem = (IEnergizedItem)itemStack.func_77973_b();
			energizedItem.setEnergy(itemStack, ((IStrictEnergyStorage)tileEntity).getEnergy());
		}

		if(tileEntity instanceof TileEntityContainerBlock && ((TileEntityContainerBlock)tileEntity).inventory.size() > 0)
		{
			ISustainedInventory inventory = (ISustainedInventory)itemStack.func_77973_b();
			inventory.setInventory(((ISustainedInventory)tileEntity).getInventory(), itemStack);
		}

		if(((ISustainedTank)itemStack.func_77973_b()).hasTank(itemStack))
		{
			if(tileEntity instanceof ISustainedTank)
			{
				if(((ISustainedTank)tileEntity).getFluidStack() != null)
				{
					((ISustainedTank)itemStack.func_77973_b()).setFluidStack(((ISustainedTank)tileEntity).getFluidStack(), itemStack);
				}
			}
		}

		if(tileEntity instanceof TileEntityFactory)
		{
			IFactory factoryItem = (IFactory)itemStack.func_77973_b();
			factoryItem.setRecipeType(((TileEntityFactory)tileEntity).recipeType.ordinal(), itemStack);
		}

		if (tileEntity instanceof TileEntityQuantumEntangloporter)
		{
			InventoryFrequency frequency = ((TileEntityQuantumEntangloporter) tileEntity).frequency;
			if (frequency != null)
			{
				ItemDataUtils.setCompound(itemStack, "entangleporter_frequency", frequency.getIdentity().serialise());
			}
		}

		return itemStack;
	}

	@Override
	public boolean canConnectRedstone(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side)
	{
		MachineType type = MachineType.get(getMachineBlock(), state.func_177230_c().func_176201_c(state));

		switch(type)
		{
			case LASER_AMPLIFIER:
				return true;
			default:
				return false;
		}
	}

	public ItemStack dismantleBlock(IBlockState state, World world, BlockPos pos, boolean returnBlock)
	{
		ItemStack itemStack = getPickBlock(state, null, world, pos, null);

		world.func_175698_g(pos);

		if(!returnBlock)
		{
			float motion = 0.7F;
			double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

			EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, itemStack);

			world.func_72838_d(entityItem);
		}

		return itemStack;
	}

	@Override
	public AxisAlignedBB func_185496_a(IBlockState state, IBlockAccess world, BlockPos pos)
	{
		MachineType type = MachineType.get(getMachineBlock(), state.func_177230_c().func_176201_c(state));
		TileEntity tile = MekanismUtils.getTileEntitySafe(world, pos);
		
		if(type == null)
		{
			return super.func_185496_a(state, world, pos);
		}
		
		switch(type)
		{
			case CHARGEPAD:
				return CHARGEPAD_BOUNDS;
			case FLUID_TANK:
				return TANK_BOUNDS;
			case LASER:
				if(tile instanceof TileEntityLaser)
				{
					return MultipartUtils.rotate(LASER_BOUNDS.func_72317_d(-0.5, -0.5, -0.5), ((TileEntityLaser)tile).facing).func_72317_d(0.5, 0.5, 0.5);
				}
			case LOGISTICAL_SORTER:
				if(tile instanceof TileEntityLogisticalSorter)
				{
					return MultipartUtils.rotate(LOGISTICAL_SORTER_BOUNDS.func_72317_d(-0.5, -0.5, -0.5), ((TileEntityLogisticalSorter)tile).facing).func_72317_d(0.5, 0.5, 0.5);
				}
			default:
				return super.func_185496_a(state, world, pos);
		}
	}

	@Override
	public boolean func_149686_d(IBlockState state)
    {
        return false;
    }

	@Override
	public boolean isSideSolid(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side)
	{
		MachineType type = MachineType.get(getMachineBlock(), state.func_177230_c().func_176201_c(state));

		switch(type)
		{
			case CHARGEPAD:
			case PERSONAL_CHEST:
				return false;
			case FLUID_TANK:
				return side == EnumFacing.UP || side == EnumFacing.DOWN;
			default:
				return true;
		}
	}

	public PropertyEnum<MachineType> getTypeProperty()
	{
		return getMachineBlock().getProperty();
	}

	@Override
	public EnumFacing[] getValidRotations(World world, BlockPos pos)
	{
		TileEntity tile = world.func_175625_s(pos);
		EnumFacing[] valid = new EnumFacing[6];
		
		if(tile instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock basicTile = (TileEntityBasicBlock)tile;
			
			for(EnumFacing dir : EnumFacing.field_82609_l)
			{
				if(basicTile.canSetFacing(dir.ordinal()))
				{
					valid[dir.ordinal()] = dir;
				}
			}
		}
		
		return valid;
	}

	@Override
	public boolean rotateBlock(World world, BlockPos pos, EnumFacing axis)
	{
		TileEntity tile = world.func_175625_s(pos);
		
		if(tile instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock basicTile = (TileEntityBasicBlock)tile;
			
			if(basicTile.canSetFacing(axis.ordinal()))
			{
				basicTile.setFacing((short)axis.ordinal());
				return true;
			}
		}
		
		return false;
	}
	
	@Override
	public int func_180656_a(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side)
    {
		TileEntity tile = MekanismUtils.getTileEntitySafe(world, pos);
		
		if(tile instanceof TileEntityLaserAmplifier)
		{
			return ((TileEntityLaserAmplifier)tile).emittingRedstone ? 15 : 0;
		}
		
        return 0;
    }
}
