package mekanism.common.block;

import java.util.Random;
import java.util.UUID;

import mekanism.api.Coord4D;
import mekanism.api.energy.IEnergizedItem;
import mekanism.api.energy.IStrictEnergyStorage;
import mekanism.common.Mekanism;
import mekanism.common.Tier.BaseTier;
import mekanism.common.base.IActiveState;
import mekanism.common.base.IBoundingBlock;
import mekanism.common.base.ITierItem;
import mekanism.common.block.states.BlockStateBasic;
import mekanism.common.block.states.BlockStateBasic.BasicBlock;
import mekanism.common.block.states.BlockStateBasic.BasicBlockType;
import mekanism.common.block.states.BlockStateFacing;
import mekanism.common.content.boiler.SynchronizedBoilerData;
import mekanism.common.content.tank.TankUpdateProtocol;
import mekanism.common.inventory.InventoryBin;
import mekanism.common.item.ItemBlockBasic;
import mekanism.common.multiblock.IMultiblock;
import mekanism.common.multiblock.IStructuralMultiblock;
import mekanism.common.tile.TileEntityBin;
import mekanism.common.tile.TileEntityDynamicTank;
import mekanism.common.tile.TileEntityInductionCell;
import mekanism.common.tile.TileEntityInductionPort;
import mekanism.common.tile.TileEntityInductionProvider;
import mekanism.common.tile.TileEntityMultiblock;
import mekanism.common.tile.TileEntitySecurityDesk;
import mekanism.common.tile.TileEntitySuperheatingElement;
import mekanism.common.tile.TileEntityThermalEvaporationController;
import mekanism.common.tile.prefab.TileEntityBasicBlock;
import mekanism.common.util.FluidContainerUtils;
import mekanism.common.util.MekanismUtils;
import mekanism.common.util.SecurityUtils;
import mekanism.common.util.StackUtils;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLiving.SpawnPlacementType;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.init.SoundEvents;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.SoundCategory;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.EnumSkyBlock;
import net.minecraft.world.Explosion;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidUtil;
import net.minecraftforge.fluids.capability.IFluidHandlerItem;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import buildcraft.api.tools.IToolWrench;

/**
 * Block class for handling multiple metal block IDs.
 * 0:0: Osmium Block
 * 0:1: Bronze Block
 * 0:2: Refined Obsidian
 * 0:3: Charcoal Block
 * 0:4: Refined Glowstone
 * 0:5: Steel Block
 * 0:6: Bin
 * 0:7: Teleporter Frame
 * 0:8: Steel Casing
 * 0:9: Dynamic Tank
 * 0:10: Structural Glass
 * 0:11: Dynamic Valve
 * 0:12: Copper Block
 * 0:13: Tin Block
 * 0:14: Thermal Evaporation Controller
 * 0:15: Thermal Evaporation Valve
 * 1:0: Thermal Evaporation Block
 * 1:1: Induction Casing
 * 1:2: Induction Port
 * 1:3: Induction Cell
 * 1:4: Induction Provider
 * 1:5: Superheating Element
 * 1:6: Pressure Disperser
 * 1:7: Boiler Casing
 * 1:8: Boiler Valve
 * 1:9: Security Desk
 * @author AidanBrady
 *
 */
public abstract class BlockBasic extends Block
{
	public BlockBasic()
	{
		super(Material.field_151573_f);
		func_149711_c(5F);
		func_149752_b(20F);
		func_149647_a(Mekanism.tabMekanism);
	}

	public static BlockBasic getBlockBasic(BasicBlock block)
	{
		return new BlockBasic() {
			@Override
			public BasicBlock getBasicBlock()
			{
				return block;
			}
		};
	}

	public abstract BasicBlock getBasicBlock();

	@Override
	public BlockStateContainer func_180661_e()
	{
		return new BlockStateBasic(this, getTypeProperty());
	}

	@Override
	public IBlockState func_176203_a(int meta)
	{
		BlockStateBasic.BasicBlockType type = BlockStateBasic.BasicBlockType.get(getBasicBlock(), meta&0xF);

		return func_176223_P().func_177226_a(getTypeProperty(), type);
	}

	@Override
	public int func_176201_c(IBlockState state)
	{
		BlockStateBasic.BasicBlockType type = state.func_177229_b(getTypeProperty());
		return type.meta;
	}

	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess worldIn, BlockPos pos)
	{
		TileEntity tile = MekanismUtils.getTileEntitySafe(worldIn, pos);

		if(tile instanceof TileEntityBasicBlock && ((TileEntityBasicBlock)tile).facing != null)
		{
			state = state.func_177226_a(BlockStateFacing.facingProperty, ((TileEntityBasicBlock)tile).facing);
		}
		
		if(tile instanceof IActiveState)
		{
			state = state.func_177226_a(BlockStateBasic.activeProperty, ((IActiveState)tile).getActive());
		}
		
		if(tile instanceof TileEntityInductionCell)
		{
			state = state.func_177226_a(BlockStateBasic.tierProperty, ((TileEntityInductionCell)tile).tier.getBaseTier());
		}
		
		if(tile instanceof TileEntityInductionProvider)
		{
			state = state.func_177226_a(BlockStateBasic.tierProperty, ((TileEntityInductionProvider)tile).tier.getBaseTier());
		}
		
		if(tile instanceof TileEntityBin)
		{
			state = state.func_177226_a(BlockStateBasic.tierProperty, ((TileEntityBin)tile).tier.getBaseTier());
		}
		
		if(tile instanceof TileEntityInductionPort)
		{
			state = state.func_177226_a(BlockStateBasic.activeProperty, ((TileEntityInductionPort)tile).mode);
		}
		
		if(tile instanceof TileEntitySuperheatingElement)
		{
			TileEntitySuperheatingElement element = (TileEntitySuperheatingElement)tile;
			boolean active = false;
			
			if(element.multiblockUUID != null && SynchronizedBoilerData.clientHotMap.get(element.multiblockUUID) != null)
			{
				active = SynchronizedBoilerData.clientHotMap.get(element.multiblockUUID);
			}
			
			state = state.func_177226_a(BlockStateBasic.activeProperty, active);
		}
		
		return state;
	}

	@Override
	public void func_189540_a(IBlockState state, World world, BlockPos pos, Block neighborBlock, BlockPos fromPos)
	{
		if(!world.field_72995_K)
		{
			TileEntity tileEntity = new Coord4D(pos, world).getTileEntity(world);

			if(tileEntity instanceof IMultiblock)
			{
				((IMultiblock)tileEntity).doUpdate();
			}

			if(tileEntity instanceof TileEntityBasicBlock)
			{
				((TileEntityBasicBlock)tileEntity).onNeighborChange(neighborBlock);
			}
			
			if(tileEntity instanceof IStructuralMultiblock)
			{
				((IStructuralMultiblock)tileEntity).doUpdate();
			}
		}
	}
	
	@Override
	public float getExplosionResistance(World world, BlockPos pos, Entity exploder, Explosion explosion)
    {
		IBlockState state = world.func_180495_p(pos);
		BasicBlockType type = BasicBlockType.get(getBasicBlock(), state.func_177230_c().func_176201_c(state));
		
		if(type == BasicBlockType.REFINED_OBSIDIAN)
		{
			return 4000F;
		}
		
		return field_149781_w;
    }

	@Override
	public int func_180651_a(IBlockState state)
	{
		return state.func_177230_c().func_176201_c(state);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_149666_a(Item item, CreativeTabs creativetabs, NonNullList<ItemStack> list)
	{
		for(BasicBlockType type : BasicBlockType.values())
		{
			if(type.blockType == getBasicBlock())
			{
				switch(type)
				{
					case INDUCTION_CELL:
					case INDUCTION_PROVIDER:
					case BIN:
						for(BaseTier tier : BaseTier.values())
						{
							if(type == BasicBlockType.BIN || tier.isObtainable())
							{
								ItemStack stack = new ItemStack(item, 1, type.meta);
								((ItemBlockBasic)stack.func_77973_b()).setBaseTier(stack, tier);
								list.add(stack);
							}
						}
						
						break;
					default:
						list.add(new ItemStack(item, 1, type.meta));
				}
			}
		}
	}

	@Override
	public boolean canCreatureSpawn(IBlockState state, IBlockAccess world, BlockPos pos, SpawnPlacementType type)
	{
		int meta = state.func_177230_c().func_176201_c(state);

		switch(getBasicBlock())
		{
			case BASIC_BLOCK_1:
				switch(meta)
				{
					case 10:
						return false;
					case 9:
					case 11:
						TileEntityDynamicTank tileEntity = (TileEntityDynamicTank) MekanismUtils.getTileEntitySafe(world, pos);

						if(tileEntity != null)
						{
							if(FMLCommonHandler.instance().getEffectiveSide() == Side.SERVER)
							{
								if(tileEntity.structure != null)
								{
									return false;
								}
							} 
							else {
								if(tileEntity.clientHasStructure)
								{
									return false;
								}
							}
						}
					default:
						return super.canCreatureSpawn(state, world, pos, type);
				}
			case BASIC_BLOCK_2:
				switch(meta)
				{
					case 1:
					case 2:
					case 7:
					case 8:
						TileEntityMultiblock tileEntity = (TileEntityMultiblock) MekanismUtils.getTileEntitySafe(world, pos);

						if(tileEntity != null)
						{
							if(FMLCommonHandler.instance().getEffectiveSide() == Side.SERVER)
							{
								if(tileEntity.structure != null)
								{
									return false;
								}
							} 
							else {
								if(tileEntity.clientHasStructure)
								{
									return false;
								}
							}
						}
					default:
						return super.canCreatureSpawn(state, world, pos, type);
				}
			default:
				return super.canCreatureSpawn(state, world, pos, type);
		}
	}

	@Override
	public void func_180649_a(World world, BlockPos pos, EntityPlayer player)
	{
		BasicBlockType type = BasicBlockType.get(world.func_180495_p(pos));

		if(!world.field_72995_K && type == BasicBlockType.BIN)
		{
			TileEntityBin bin = (TileEntityBin)world.func_175625_s(pos);
			RayTraceResult mop = MekanismUtils.rayTrace(world, player);

			if(mop != null && mop.field_178784_b == bin.facing)
			{
				if(!bin.bottomStack.func_190926_b())
				{
					ItemStack stack;
					if(player.func_70093_af())
					{
						stack = bin.remove(1).func_77946_l();
					}
					else
					{
						stack = bin.removeStack().func_77946_l();
					}

					if(!player.field_71071_by.func_70441_a(stack))
					{
						BlockPos dropPos = pos.func_177972_a(bin.facing);
						Entity item = new EntityItem(world, dropPos.func_177958_n() + .5f, dropPos.func_177956_o() + .3f, dropPos.func_177952_p() + .5f, stack);
						item.func_70024_g(-item.field_70159_w, -item.field_70181_x, -item.field_70179_y);
						world.func_72838_d(item);
					}
					else
					{
						world.func_184148_a(null, pos.func_177958_n() + .5f, pos.func_177956_o() + .5f, pos.func_177952_p() + .5f,
								SoundEvents.field_187638_cR, SoundCategory.PLAYERS, 0.2F, ((world.field_73012_v.nextFloat() - world.field_73012_v.nextFloat()) * 0.7F + 1.0F) * 2.0F);
					}
				}
			}
		}
	}

	@Override
	public boolean func_180639_a(World world, BlockPos pos, IBlockState state, EntityPlayer entityplayer, EnumHand hand, EnumFacing side, float hitX, float hitY, float hitZ)
	{
		BasicBlockType type = BasicBlockType.get(state);
		TileEntity tile = world.func_175625_s(pos);
		ItemStack stack = entityplayer.func_184586_b(hand);

		if(type == BasicBlockType.REFINED_OBSIDIAN)
		{
			if(entityplayer.func_70093_af())
			{
				entityplayer.openGui(Mekanism.instance, 1, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
				return true;
			}
		}

		if(tile instanceof TileEntityThermalEvaporationController)
		{
			if(!entityplayer.func_70093_af())
			{
				if(!world.field_72995_K)
				{
					entityplayer.openGui(Mekanism.instance, 33, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
				}
				
				return true;
			}
		}
		else if(tile instanceof TileEntitySecurityDesk)
		{
			UUID ownerUUID = ((TileEntitySecurityDesk)tile).ownerUUID;
			
			if(!entityplayer.func_70093_af())
			{
				if(!world.field_72995_K)
				{
					if(ownerUUID == null || entityplayer.func_110124_au().equals(ownerUUID))
					{
						entityplayer.openGui(Mekanism.instance, 57, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
					}
					else {
						SecurityUtils.displayNoAccess(entityplayer);
					}
				}
				
				return true;
			}
		}
		else if(tile instanceof TileEntityBin)
		{
			TileEntityBin bin = (TileEntityBin)tile;

			if(!stack.func_190926_b() && MekanismUtils.hasUsableWrench(entityplayer, pos))
			{
				if(!world.field_72995_K)
				{
					Item tool = stack.func_77973_b();
					
					if(entityplayer.func_70093_af())
					{
						dismantleBlock(state, world, pos, false);
						return true;
					}
	
					if(MekanismUtils.isBCWrench(tool))
					{
						((IToolWrench)tool).wrenchUsed(entityplayer, hand, stack, new RayTraceResult(new Vec3d(hitX, hitY, hitZ), side, pos));
					}
	
					int change = bin.facing.func_176746_e().ordinal();
	
					bin.setFacing((short)change);
					world.func_175685_c(pos, this, true);
				}
				
				return true;
			}

			if(!world.field_72995_K)
			{
				if(bin.getItemCount() < bin.tier.storage)
				{
					if(bin.addTicks == 0)
					{
						if(!stack.func_190926_b())
						{
							ItemStack remain = bin.add(stack);
							entityplayer.func_184611_a(hand, remain);
							bin.addTicks = 5;
						}
					}
					else if(bin.addTicks > 0 && bin.getItemCount() > 0)
					{
						NonNullList<ItemStack> inv = entityplayer.field_71071_by.field_70462_a;
	
						for(int i = 0; i < inv.size(); i++)
						{
							if(bin.getItemCount() == bin.tier.storage)
							{
								break;
							}
	
							if(!inv.get(i).func_190926_b())
							{
								ItemStack remain = bin.add(inv.get(i));
								inv.set(i, remain);
								bin.addTicks = 5;
							}
	
							((EntityPlayerMP)entityplayer).func_71120_a(entityplayer.field_71070_bA);
						}
					}
				}
			}

			return true;
		}
		else if(tile instanceof IMultiblock)
		{
			if(world.field_72995_K)
			{
				return true;
			}
			
			return ((IMultiblock)world.func_175625_s(pos)).onActivate(entityplayer, hand, stack);
		}
		else if(tile instanceof IStructuralMultiblock)
		{
			if(world.field_72995_K)
			{
				return true;
			}
			
			return ((IStructuralMultiblock)world.func_175625_s(pos)).onActivate(entityplayer, hand, stack);
		}

		return false;
	}

	@Override
	public boolean isSideSolid(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side)
	{
		return BasicBlockType.get(state) != BasicBlockType.STRUCTURAL_GLASS;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public BlockRenderLayer func_180664_k()
	{
		return BlockRenderLayer.CUTOUT;
	}

	public static boolean manageInventory(EntityPlayer player, TileEntityDynamicTank tileEntity, EnumHand hand, ItemStack itemStack)
	{
		if(tileEntity.structure == null)
		{
			return false;
		}
		
		ItemStack copyStack = StackUtils.size(itemStack, 1);
		
		if(FluidContainerUtils.isFluidContainer(itemStack))
		{
			IFluidHandlerItem handler = FluidUtil.getFluidHandler(copyStack);
			
			if(FluidUtil.getFluidContained(copyStack) == null)
			{
				if(tileEntity.structure.fluidStored != null)
				{
					int filled = handler.fill(tileEntity.structure.fluidStored, !player.field_71075_bZ.field_75098_d);
					copyStack = handler.getContainer();
					
					if(filled > 0)
					{
						if(player.field_71075_bZ.field_75098_d)
						{
							tileEntity.structure.fluidStored.amount -= filled;
						}
						else if(itemStack.func_190916_E() == 1)
						{
							tileEntity.structure.fluidStored.amount -= filled;
							player.func_184611_a(hand, copyStack);
						}
						else if(itemStack.func_190916_E() > 1 && player.field_71071_by.func_70441_a(copyStack))
						{
							tileEntity.structure.fluidStored.amount -= filled;
							itemStack.func_190918_g(1);
						}
						
						if(tileEntity.structure.fluidStored.amount == 0)
						{
							tileEntity.structure.fluidStored = null;
						}
						
						return true;
					}
				}
			}
			else {
				FluidStack itemFluid = FluidUtil.getFluidContained(copyStack);
				int stored = tileEntity.structure.fluidStored != null ? tileEntity.structure.fluidStored.amount : 0;
				int needed = (tileEntity.structure.volume*TankUpdateProtocol.FLUID_PER_TANK)-stored;
				
				if(tileEntity.structure.fluidStored != null && !tileEntity.structure.fluidStored.isFluidEqual(itemFluid))
				{
					return false;
				}
				
				boolean filled = false;
				FluidStack drained = handler.drain(needed, !player.field_71075_bZ.field_75098_d);
				copyStack = handler.getContainer();
				
				if(copyStack.func_190916_E() == 0)
				{
					copyStack = ItemStack.field_190927_a;
				}
				
				if(drained != null)
				{
					if(player.field_71075_bZ.field_75098_d)
					{
						filled = true;
					}
					else {
						if(!copyStack.func_190926_b())
						{
							if(itemStack.func_190916_E() == 1)
							{
								player.func_184611_a(hand, copyStack);
								filled = true;
							}
							else {
								if(player.field_71071_by.func_70441_a(copyStack))
								{
									itemStack.func_190918_g(1);
	
									filled = true;
								}
							}
						}
						else {
							itemStack.func_190918_g(1);
	
							if(itemStack.func_190916_E() == 0)
							{
								player.func_184611_a(hand, ItemStack.field_190927_a);
							}
	
							filled = true;
						}
					}
	
					if(filled)
					{
						if(tileEntity.structure.fluidStored == null)
						{
							tileEntity.structure.fluidStored = drained;
						}
						else {
							tileEntity.structure.fluidStored.amount += drained.amount;
						}
						
						return true;
					}
				}
			}
		}

		return false;
	}

	@Override
	public boolean func_149662_c(IBlockState state)
	{
		return false;
	}
	
	@Override
	public boolean func_149686_d(IBlockState state)
	{
		return false;
	}

	@Override
	public EnumBlockRenderType func_149645_b(IBlockState state)
	{
		return EnumBlockRenderType.MODEL;
	}

	@Override
	public int getLightValue(IBlockState state, IBlockAccess world, BlockPos pos)
	{
		TileEntity tileEntity = MekanismUtils.getTileEntitySafe(world, pos);
		int metadata = state.func_177230_c().func_176201_c(state);

		if(tileEntity instanceof IActiveState)
		{
			if(((IActiveState)tileEntity).getActive() && ((IActiveState)tileEntity).lightUpdate())
			{
				return 15;
			}
		}

		if(getBasicBlock() == BasicBlock.BASIC_BLOCK_1)
		{
			switch(metadata)
			{
				case 2:
					return 8;
				case 4:
					return 15;
				case 7:
					return 12;
			}
		}
		else if(getBasicBlock() == BasicBlock.BASIC_BLOCK_2)
		{
			if(metadata == 5 && tileEntity instanceof TileEntitySuperheatingElement)
			{
				TileEntitySuperheatingElement element = (TileEntitySuperheatingElement)tileEntity;
				
				if(element.multiblockUUID != null && SynchronizedBoilerData.clientHotMap.get(element.multiblockUUID) != null)
				{
					return SynchronizedBoilerData.clientHotMap.get(element.multiblockUUID) ? 15 : 0;
				}
				
				return 0;
			}
		}

		return 0;
	}

	@Override
	public boolean hasTileEntity(IBlockState state)
	{
		BasicBlockType type = BasicBlockType.get(state);
		
		return type != null && type.tileEntityClass != null;
	}

	@Override
	public TileEntity createTileEntity(World world, IBlockState state)
	{
		if(BasicBlockType.get(state) == null)
		{
			return null;
		}
		
		return BasicBlockType.get(state).create();
	}

	@Override
	public void func_180633_a(World world, BlockPos pos, IBlockState state, EntityLivingBase placer, ItemStack stack)
	{
		if(world.func_175625_s(pos) instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock tileEntity = (TileEntityBasicBlock)world.func_175625_s(pos);
			int side = MathHelper.func_76128_c((placer.field_70177_z * 4.0F / 360.0F) + 0.5D) & 3;
			int height = Math.round(placer.field_70125_A);
			int change = 3;
			
			if(tileEntity.canSetFacing(0) && tileEntity.canSetFacing(1))
			{
				if(height >= 65)
				{
					change = 1;
				}
				else if(height <= -65)
				{
					change = 0;
				}
			}

			if(change != 0 && change != 1)
			{
				switch(side)
				{
					case 0: change = 2; break;
					case 1: change = 5; break;
					case 2: change = 3; break;
					case 3: change = 4; break;
				}
			}

			tileEntity.setFacing((short)change);
			tileEntity.redstone = world.func_175687_A(pos) > 0;
			
			if(tileEntity instanceof TileEntitySecurityDesk)
			{
				((TileEntitySecurityDesk)tileEntity).ownerUUID = placer.func_110124_au();
			}

			if(tileEntity instanceof IBoundingBlock)
			{
				((IBoundingBlock)tileEntity).onPlace();
			}
		}

		world.func_175704_b(pos, pos.func_177982_a(1,1,1));
		world.func_180500_c(EnumSkyBlock.BLOCK, pos);
	    world.func_180500_c(EnumSkyBlock.SKY, pos);

		if(!world.field_72995_K && world.func_175625_s(pos) != null)
		{
			TileEntity tileEntity = world.func_175625_s(pos);

			if(tileEntity instanceof IMultiblock)
			{
				((IMultiblock)tileEntity).doUpdate();
			}
			
			if(tileEntity instanceof IStructuralMultiblock)
			{
				((IStructuralMultiblock)tileEntity).doUpdate();
			}
		}
	}
	
	@Override
	public void func_180663_b(World world, BlockPos pos, IBlockState state)
	{
		TileEntity tileEntity = world.func_175625_s(pos);

		if(tileEntity instanceof IBoundingBlock)
		{
			((IBoundingBlock)tileEntity).onBreak();
		}

		super.func_180663_b(world, pos, state);
	}

	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player)
	{
		BasicBlockType type = BasicBlockType.get(state);
		ItemStack ret = new ItemStack(this, 1, state.func_177230_c().func_176201_c(state));

		if(type == BasicBlockType.BIN)
		{
			TileEntityBin tileEntity = (TileEntityBin)world.func_175625_s(pos);
			InventoryBin inv = new InventoryBin(ret);

			((ITierItem)ret.func_77973_b()).setBaseTier(ret, tileEntity.tier.getBaseTier());
			inv.setItemCount(tileEntity.getItemCount());
			
			if(tileEntity.getItemCount() > 0)
			{
				inv.setItemType(tileEntity.itemType);
			}
		}
		else if(type == BasicBlockType.INDUCTION_CELL)
		{
			TileEntityInductionCell tileEntity = (TileEntityInductionCell)world.func_175625_s(pos);
			((ItemBlockBasic)ret.func_77973_b()).setBaseTier(ret, tileEntity.tier.getBaseTier());
		}
		else if(type == BasicBlockType.INDUCTION_PROVIDER)
		{
			TileEntityInductionProvider tileEntity = (TileEntityInductionProvider)world.func_175625_s(pos);
			((ItemBlockBasic)ret.func_77973_b()).setBaseTier(ret, tileEntity.tier.getBaseTier());
		}
		
		TileEntity tileEntity = world.func_175625_s(pos);
		
		if(tileEntity instanceof IStrictEnergyStorage)
		{
			IEnergizedItem energizedItem = (IEnergizedItem)ret.func_77973_b();
			energizedItem.setEnergy(ret, ((IStrictEnergyStorage)tileEntity).getEnergy());
		}

		return ret;
	}

	@Override
	public Item func_180660_a(IBlockState state, Random random, int fortune)
	{
		return null;
	}

	@Override
	public boolean removedByPlayer(IBlockState state, World world, BlockPos pos, EntityPlayer player, boolean willHarvest)
	{
		if(!player.field_71075_bZ.field_75098_d && !world.field_72995_K && willHarvest)
		{
			float motion = 0.7F;
			double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

			EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, getPickBlock(state, null, world, pos, player));

			world.func_72838_d(entityItem);
		}

		return world.func_175698_g(pos);
	}

	public ItemStack dismantleBlock(IBlockState state, World world, BlockPos pos, boolean returnBlock)
	{
		ItemStack itemStack = getPickBlock(state, null, world, pos, null);

		world.func_175698_g(pos);

		if(!returnBlock)
		{
			float motion = 0.7F;
			double motionX = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionY = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;
			double motionZ = (world.field_73012_v.nextFloat() * motion) + (1.0F - motion) * 0.5D;

			EntityItem entityItem = new EntityItem(world, pos.func_177958_n() + motionX, pos.func_177956_o() + motionY, pos.func_177952_p() + motionZ, itemStack);

			world.func_72838_d(entityItem);
		}

		return itemStack;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public boolean func_176225_a(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing side)
	{
		if(BasicBlockType.get(state) == BasicBlockType.STRUCTURAL_GLASS && BasicBlockType.get(world.func_180495_p(pos.func_177972_a(side))) == BasicBlockType.STRUCTURAL_GLASS)
		{
			return false;
		}
		
		return super.func_176225_a(state, world, pos, side);
	}

	@Override
	public EnumFacing[] getValidRotations(World world, BlockPos pos)
	{
		TileEntity tile = world.func_175625_s(pos);
		EnumFacing[] valid = new EnumFacing[6];
		
		if(tile instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock basicTile = (TileEntityBasicBlock)tile;
			
			for(EnumFacing dir : EnumFacing.field_82609_l)
			{
				if(basicTile.canSetFacing(dir.ordinal()))
				{
					valid[dir.ordinal()] = dir;
				}
			}
		}
		
		return valid;
	}

	@Override
	public boolean rotateBlock(World world, BlockPos pos, EnumFacing axis)
	{
		TileEntity tile = world.func_175625_s(pos);
		
		if(tile instanceof TileEntityBasicBlock)
		{
			TileEntityBasicBlock basicTile = (TileEntityBasicBlock)tile;
			
			if(basicTile.canSetFacing(axis.ordinal()))
			{
				basicTile.setFacing((short)axis.ordinal());
				return true;
			}
		}
		
		return false;
	}

	public PropertyEnum<BasicBlockType> getTypeProperty()
	{
		return getBasicBlock().getProperty();
	}

	@Override
	public boolean isBeaconBase(IBlockAccess world, BlockPos pos, BlockPos beacon) 
	{
		return BasicBlockType.get(world.func_180495_p(pos)).isBeaconBase;
	}
}
