package mekanism.client.jei.gas;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import mekanism.api.gas.Gas;
import mekanism.api.gas.GasStack;
import mekanism.common.util.LangUtils;
import mezz.jei.api.gui.IDrawable;
import mezz.jei.api.ingredients.IIngredientRenderer;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.VertexBuffer;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.text.TextFormatting;
import net.minecraftforge.fluids.Fluid;

public class GasStackRenderer implements IIngredientRenderer<GasStack>
{
	private static final int TEX_WIDTH = 16;
	private static final int TEX_HEIGHT = 16;
	private static final int MIN_FLUID_HEIGHT = 1; // ensure tiny amounts of gas are still visible

	private final int capacityMb;
	private final TooltipMode tooltipMode;
	private final int width;
	private final int height;
	@Nullable
	private final IDrawable overlay;

	enum TooltipMode 
	{
		SHOW_AMOUNT,
		SHOW_AMOUNT_AND_CAPACITY,
		ITEM_LIST
	}

	public GasStackRenderer()
	{
		this(Fluid.BUCKET_VOLUME, TooltipMode.ITEM_LIST, TEX_WIDTH, TEX_HEIGHT, null);
	}

	public GasStackRenderer(int capacityMb, boolean showCapacity, int width, int height, @Nullable IDrawable overlay) 
	{
		this(capacityMb, showCapacity ? TooltipMode.SHOW_AMOUNT_AND_CAPACITY : TooltipMode.SHOW_AMOUNT, width, height, overlay);
	}

	public GasStackRenderer(int capacityMb, TooltipMode tooltipMode, int width, int height, @Nullable IDrawable overlay) 
	{
		this.capacityMb = capacityMb;
		this.tooltipMode = tooltipMode;
		this.width = width;
		this.height = height;
		this.overlay = overlay;
	}

	@Override
	public void render(Minecraft minecraft, final int xPosition, final int yPosition, @Nullable GasStack gasStack) 
	{
		GlStateManager.func_179147_l();
		GlStateManager.func_179141_d();

		drawGas(minecraft, xPosition, yPosition, gasStack);

		GlStateManager.func_179131_c(1, 1, 1, 1);

		if(overlay != null) 
		{
			GlStateManager.func_179094_E();
			GlStateManager.func_179109_b(0, 0, 200);
			overlay.draw(minecraft, xPosition, yPosition);
			GlStateManager.func_179121_F();
		}

		GlStateManager.func_179118_c();
		GlStateManager.func_179084_k();
	}

	private void drawGas(Minecraft minecraft, final int xPosition, final int yPosition, @Nullable GasStack gasStack)
	{
		if(gasStack == null) 
		{
			return;
		}
		
		Gas gas = gasStack.getGas();
		
		if(gas == null)
		{
			return;
		}

		TextureAtlasSprite gasStillSprite = getStillGasSprite(minecraft, gas);

		int gasColor = 0xFFFFFFFF;

		int scaledAmount = (gasStack.amount * height) / capacityMb;
		
		if(gasStack.amount > 0 && scaledAmount < MIN_FLUID_HEIGHT) 
		{
			scaledAmount = MIN_FLUID_HEIGHT;
		}
		
		if(scaledAmount > height)
		{
			scaledAmount = height;
		}

		drawTiledSprite(minecraft, xPosition, yPosition, width, height, gasColor, scaledAmount, gasStillSprite);
	}

	private void drawTiledSprite(Minecraft minecraft, final int xPosition, final int yPosition, final int tiledWidth, final int tiledHeight, int color, int scaledAmount, TextureAtlasSprite sprite)
	{
		minecraft.field_71446_o.func_110577_a(TextureMap.field_110575_b);
		setGLColorFromInt(color);

		final int xTileCount = tiledWidth / TEX_WIDTH;
		final int xRemainder = tiledWidth - (xTileCount * TEX_WIDTH);
		final int yTileCount = scaledAmount / TEX_HEIGHT;
		final int yRemainder = scaledAmount - (yTileCount * TEX_HEIGHT);

		final int yStart = yPosition + tiledHeight;

		for(int xTile = 0; xTile <= xTileCount; xTile++) 
		{
			for(int yTile = 0; yTile <= yTileCount; yTile++) 
			{
				int width = (xTile == xTileCount) ? xRemainder : TEX_WIDTH;
				int height = (yTile == yTileCount) ? yRemainder : TEX_HEIGHT;
				int x = xPosition + (xTile * TEX_WIDTH);
				int y = yStart - ((yTile + 1) * TEX_HEIGHT);
				
				if(width > 0 && height > 0) 
				{
					int maskTop = TEX_HEIGHT - height;
					int maskRight = TEX_WIDTH - width;

					drawTextureWithMasking(x, y, sprite, maskTop, maskRight, 100);
				}
			}
		}
	}

	private static TextureAtlasSprite getStillGasSprite(Minecraft minecraft, Gas gas) 
	{
		TextureMap textureMapBlocks = minecraft.func_147117_R();
		ResourceLocation gasStill = gas.getIcon();
		TextureAtlasSprite gasStillSprite = null;
		
		if(gasStill != null)
		{
			gasStillSprite = textureMapBlocks.getTextureExtry(gasStill.toString());
		}
		
		if(gasStillSprite == null) 
		{
			gasStillSprite = textureMapBlocks.func_174944_f();
		}
		
		return gasStillSprite;
	}

	private static void setGLColorFromInt(int color) 
	{
		float red = (color >> 16 & 0xFF) / 255.0F;
		float green = (color >> 8 & 0xFF) / 255.0F;
		float blue = (color & 0xFF) / 255.0F;

		GlStateManager.func_179131_c(red, green, blue, 1.0F);
	}

	private static void drawTextureWithMasking(double xCoord, double yCoord, TextureAtlasSprite textureSprite, int maskTop, int maskRight, double zLevel)
	{
		double uMin = (double) textureSprite.func_94209_e();
		double uMax = (double) textureSprite.func_94212_f();
		double vMin = (double) textureSprite.func_94206_g();
		double vMax = (double) textureSprite.func_94210_h();
		uMax = uMax - (maskRight / 16.0 * (uMax - uMin));
		vMax = vMax - (maskTop / 16.0 * (vMax - vMin));

		Tessellator tessellator = Tessellator.func_178181_a();
		VertexBuffer vertexBuffer = tessellator.func_178180_c();
		vertexBuffer.func_181668_a(7, DefaultVertexFormats.field_181707_g);
		vertexBuffer.func_181662_b(xCoord, yCoord + 16, zLevel).func_187315_a(uMin, vMax).func_181675_d();
		vertexBuffer.func_181662_b(xCoord + 16 - maskRight, yCoord + 16, zLevel).func_187315_a(uMax, vMax).func_181675_d();
		vertexBuffer.func_181662_b(xCoord + 16 - maskRight, yCoord + maskTop, zLevel).func_187315_a(uMax, vMin).func_181675_d();
		vertexBuffer.func_181662_b(xCoord, yCoord + maskTop, zLevel).func_187315_a(uMin, vMin).func_181675_d();
		tessellator.func_78381_a();
	}

	@Override
	public List<String> getTooltip(Minecraft minecraft, GasStack gasStack) 
	{
		List<String> tooltip = new ArrayList<String>();
		Gas gasType = gasStack.getGas();
		
		if(gasType == null) 
		{
			return tooltip;
		}

		String gasName = gasType.getLocalizedName();
		tooltip.add(gasName);

		if(tooltipMode == TooltipMode.SHOW_AMOUNT_AND_CAPACITY) 
		{
			String amount = LangUtils.localizeWithFormat("jei.tooltip.liquid.amount.with.capacity", gasStack.amount, capacityMb);
			tooltip.add(TextFormatting.GRAY + amount);
		} 
		else if (tooltipMode == TooltipMode.SHOW_AMOUNT)
		{
			String amount = LangUtils.localizeWithFormat("jei.tooltip.liquid.amount", gasStack.amount);
			tooltip.add(TextFormatting.GRAY + amount);
		}

		return tooltip;
	}

	@Override
	public FontRenderer getFontRenderer(Minecraft minecraft, GasStack gasStack) {
		return minecraft.field_71466_p;
	}
}
