package mekanism.api;

import net.minecraft.entity.Entity;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.math.Vec3i;

/**
 * Pos3D - a way of performing operations on objects in a three dimensional environment.
 * @author aidancbrady
 *
 */
public class Pos3D extends Vec3d
{
	public Pos3D()
	{
		this(0, 0, 0);
	}

	public Pos3D(Vec3d vec)
	{
		super(vec.field_72450_a, vec.field_72448_b, vec.field_72449_c);
	}
	
	public Pos3D(Coord4D coord)
	{
		super(coord.xCoord, coord.yCoord, coord.zCoord);
	}

	public Pos3D(Vec3i vec)
	{
		super(vec);
	}

	public Pos3D(RayTraceResult mop)
	{
		this(mop.func_178782_a());
	}

	public Pos3D(double x, double y, double z)
	{
		super(x, y, z);
	}

	/**
	 * Creates a Pos3D with an entity's posX, posY, and posZ values.
	 * @param entity - entity to create the Pos3D from
	 */
	public Pos3D(Entity entity)
	{
		this(entity.field_70165_t, entity.field_70163_u, entity.field_70161_v);
	}

	/**
	 * Creates a Pos3D with a TileEntity's xCoord, yCoord and zCoord values.
	 * @param tileEntity - TileEntity to create the Pos3D from
	 */
	public Pos3D(TileEntity tileEntity)
	{
		this(tileEntity.func_174877_v());
	}
	
	/**
	 * Returns a new Pos3D from a tag compound.
	 * @param tag - tag compound to read from
	 * @return the Pos3D from the tag compound
	 */
    public static Pos3D read(NBTTagCompound tag)
    {
        return new Pos3D(tag.func_74769_h("x"), tag.func_74769_h("y"), tag.func_74769_h("z"));
    }
    
    /**
	 * Writes this Pos3D's data to an NBTTagCompound.
	 * @param nbtTags - tag compound to write to
	 * @return the tag compound with this Pos3D's data
	 */
	public NBTTagCompound write(NBTTagCompound nbtTags)
	{
		nbtTags.func_74780_a("x", field_72450_a);
		nbtTags.func_74780_a("y", field_72448_b);
		nbtTags.func_74780_a("z", field_72449_c);

		return nbtTags;
	}

	/**
	 * Creates and returns a Pos3D with values representing the difference between this and the Pos3D in the parameters.
	 * @param vec - Vec3 to subtract
	 * @return difference of the two Pos3Ds
	 */
	public Pos3D diff(Vec3d vec)
	{
		return new Pos3D(field_72450_a-vec.field_72450_a, field_72448_b-vec.field_72448_b, field_72449_c-vec.field_72449_c);
	}

	/**
	 * Creates a new Pos3D from the motion of an entity.
	 * @param entity
	 * @return Pos3D representing the motion of the given entity
	 */
	public static Pos3D fromMotion(Entity entity)
	{
		return new Pos3D(entity.field_70159_w, entity.field_70181_x, entity.field_70179_y);
	}

    /**
     * Creates a new Coord4D representing this Pos3D in the provided dimension.
     * @param dimensionId - the dimension this Pos3D is in
     * @return Coord4D representing this Pos3D
     */
    public Coord4D getCoord(int dimensionId)
    {
        return new Coord4D((int)field_72450_a, (int)field_72448_b, (int)field_72449_c, dimensionId);
    }

	/**
	 * Centres a block-derived Pos3D
	 */
	public Pos3D centre()
	{
		return translate(0.5, 0.5, 0.5);
	}

	/**
	 * Translates this Pos3D by the defined values.
	 * @param x - amount to translate on the x axis
	 * @param y - amount to translate on the y axis
	 * @param z - amount to translate on the z axis
	 * @return the translated Pos3D
	 */
	public Pos3D translate(double x, double y, double z)
	{
		return new Pos3D(field_72450_a + x, field_72448_b + y, field_72449_c + z);
	}

	/**
	 * Performs the same operation as translate(x, y, z), but with a Pos3D value instead.
	 * @param pos - Pos3D value to translate by
	 * @return translated Pos3D
	 */
	public Pos3D translate(Vec3d pos)
	{
		return translate(pos.field_72450_a, pos.field_72448_b, pos.field_72449_c);
	}

	/**
	 * Performs the same operation as translate(x, y, z), but by a set amount in a EnumFacing
	 */
	public Pos3D translate(EnumFacing direction, double amount)
	{
		return translate(direction.func_176730_m().func_177958_n() * amount, direction.func_176730_m().func_177956_o() * amount, direction.func_176730_m().func_177952_p() * amount);
	}

	/**
	 * Performs the same operation as translate(x, y, z), but by a set amount in a EnumFacing
	 */
	public Pos3D translateExcludingSide(EnumFacing direction, double amount)
	{
		double xPos = field_72450_a, yPos = field_72448_b, zPos = field_72449_c;
		if(direction.func_176740_k() != Axis.X) xPos += amount;
		if(direction.func_176740_k() != Axis.Y) yPos += amount;
		if(direction.func_176740_k() != Axis.Z) zPos += amount;

		return new Pos3D(xPos, yPos, zPos);
	}

	/**
	 * Returns the distance between this and the defined Pos3D.
	 * @param pos - the Pos3D to find the distance to
	 * @return the distance between this and the defined Pos3D
	 */
	public double distance(Vec3d pos)
	{
		double subX = field_72450_a - pos.field_72450_a;
		double subY = field_72448_b - pos.field_72448_b;
		double subZ = field_72449_c - pos.field_72449_c;
		return MathHelper.func_76133_a(subX * subX + subY * subY + subZ * subZ);
	}

	/**
	 * Rotates this Pos3D by the defined yaw value.
	 * @param yaw - yaw to rotate by
	 * @return rotated Pos3D
	 */
	public Pos3D func_178785_b(float yaw)
	{
		double yawRadians = Math.toRadians(yaw);

		double xPos = field_72450_a;
		double zPos = field_72449_c;

		if(yaw != 0)
		{
			xPos = field_72450_a * Math.cos(yawRadians) - field_72449_c * Math.sin(yawRadians);
			zPos = field_72449_c * Math.cos(yawRadians) + field_72450_a * Math.sin(yawRadians);
		}

		return new Pos3D(xPos, field_72448_b, zPos);
	}
	
	@Override
	public Pos3D func_178789_a(float pitch)
	{
		double pitchRadians = Math.toRadians(pitch);
		
		double yPos = field_72448_b;
		double zPos = field_72449_c;
		
		if(pitch != 0)
		{
			yPos = field_72448_b * Math.cos(pitchRadians) - field_72449_c * Math.sin(pitchRadians);
			zPos = field_72449_c * Math.cos(pitchRadians) + field_72448_b * Math.sin(pitchRadians);
		}
		
		return new Pos3D(field_72450_a, yPos, zPos);
	}
	
	public Pos3D rotate(float yaw, float pitch)
	{
		return rotate(yaw, pitch, 0);
	}

    public Pos3D rotate(float yaw, float pitch, float roll)
    {
        double yawRadians = Math.toRadians(yaw);
        double pitchRadians = Math.toRadians(pitch);
        double rollRadians = Math.toRadians(roll);

        double xPos = field_72450_a;
        double yPos = field_72448_b;
        double zPos = field_72449_c;

        xPos = field_72450_a * Math.cos(yawRadians) * Math.cos(pitchRadians) + field_72449_c * (Math.cos(yawRadians) * Math.sin(pitchRadians) * Math.sin(rollRadians) - Math.sin(yawRadians) * Math.cos(rollRadians)) + field_72448_b * (Math.cos(yawRadians) * Math.sin(pitchRadians) * Math.cos(rollRadians) + Math.sin(yawRadians) * Math.sin(rollRadians));
        zPos = field_72450_a * Math.sin(yawRadians) * Math.cos(pitchRadians) + field_72449_c * (Math.sin(yawRadians) * Math.sin(pitchRadians) * Math.sin(rollRadians) + Math.cos(yawRadians) * Math.cos(rollRadians)) + field_72448_b * (Math.sin(yawRadians) * Math.sin(pitchRadians) * Math.cos(rollRadians) - Math.cos(yawRadians) * Math.sin(rollRadians));
        yPos = -field_72450_a * Math.sin(pitchRadians) + field_72449_c * Math.cos(pitchRadians) * Math.sin(rollRadians) + field_72448_b * Math.cos(pitchRadians) * Math.cos(rollRadians);
        
        return new Pos3D(xPos, yPos, zPos);
    }
	
	public Pos3D multiply(Vec3d pos)
	{
		return scale(pos.field_72450_a, pos.field_72448_b, pos.field_72449_c);
	}

	/**
	 * Scales this Pos3D by the defined x, y, an z values.
	 * @param x - x value to scale by
	 * @param y - y value to scale by
	 * @param z - z value to scale by
	 * @return scaled Pos3D
	 */
	public Pos3D scale(double x, double y, double z)
	{
		return new Pos3D(field_72450_a * x, field_72448_b * y, field_72449_c * z);
	}

	@Override
	public Pos3D func_186678_a(double scale)
	{
		return scale(scale, scale, scale);
	}
	
	public Pos3D rotate(float angle, Pos3D axis)
	{
		return translateMatrix(getRotationMatrix(angle, axis), this);
	}

	public double[] getRotationMatrix(float angle)
	{
		double[] matrix = new double[16];
		Pos3D axis = clone().func_72432_b();
		
		double x = axis.field_72450_a;
		double y = axis.field_72448_b;
		double z = axis.field_72449_c;
		
		angle *= 0.0174532925D;
		
		float cos = (float)Math.cos(angle);
		float ocos = 1.0F - cos;
		float sin = (float)Math.sin(angle);
		
		matrix[0] = (x * x * ocos + cos);
		matrix[1] = (y * x * ocos + z * sin);
		matrix[2] = (x * z * ocos - y * sin);
		matrix[4] = (x * y * ocos - z * sin);
		matrix[5] = (y * y * ocos + cos);
		matrix[6] = (y * z * ocos + x * sin);
		matrix[8] = (x * z * ocos + y * sin);
		matrix[9] = (y * z * ocos - x * sin);
		matrix[10] = (z * z * ocos + cos);
		matrix[15] = 1.0F;
		
		return matrix;
	}

	public static Pos3D translateMatrix(double[] matrix, Pos3D translation)
	{
		double x = translation.field_72450_a * matrix[0] + translation.field_72448_b * matrix[1] + translation.field_72449_c * matrix[2] + matrix[3];
		double y = translation.field_72450_a * matrix[4] + translation.field_72448_b * matrix[5] + translation.field_72449_c * matrix[6] + matrix[7];
		double z = translation.field_72450_a * matrix[8] + translation.field_72448_b * matrix[9] + translation.field_72449_c * matrix[10] + matrix[11];
		
		return new Pos3D(x, y, z);
	}

	public static double[] getRotationMatrix(float angle, Pos3D axis)
	{
		return axis.getRotationMatrix(angle);
	}
	
	public double anglePreNorm(Pos3D pos2)
	{
		return Math.acos(func_72430_b(pos2));
	}

	public static double anglePreNorm(Pos3D pos1, Pos3D pos2)
	{
		return Math.acos(pos1.clone().func_72430_b(pos2));
	}

	@Override
	public Pos3D func_72432_b() 
	{
		return new Pos3D(super.func_72432_b());
	}
	
	public Pos3D xCrossProduct()
	{
		return new Pos3D(0.0D, field_72449_c, -field_72448_b);
	}

	public Pos3D zCrossProduct()
	{
		return new Pos3D(-field_72448_b, field_72450_a, 0.0D);
	}
	
	public Pos3D getPerpendicular()
	{
		if(field_72449_c == 0)
		{
			return zCrossProduct();
		}

		return xCrossProduct();
	}
	
	public Pos3D floor()
	{
		return new Pos3D(Math.floor(field_72450_a), Math.floor(field_72448_b), Math.floor(field_72449_c));
	}

	public static AxisAlignedBB getAABB(Pos3D pos1, Pos3D pos2)
	{
		return new AxisAlignedBB(
			pos1.field_72450_a,
			pos1.field_72448_b,
			pos1.field_72449_c,
			pos2.field_72450_a,
			pos2.field_72448_b,
			pos2.field_72449_c
		);
	}

	@Override
	public Pos3D clone()
	{
		return new Pos3D(field_72450_a, field_72448_b, field_72449_c);
	}

	@Override
	public String toString()
	{
		return "[Pos3D: " + field_72450_a + ", " + field_72448_b + ", " + field_72449_c + "]";
	}

	@Override
	public boolean equals(Object obj)
	{
		return obj instanceof Vec3d &&
				((Vec3d)obj).field_72450_a == field_72450_a &&
				((Vec3d)obj).field_72450_a == field_72448_b &&
				((Vec3d)obj).field_72450_a == field_72449_c;
	}

	@Override
	public int hashCode()
	{
		int code = 1;
		code = 31 * code + new Double(field_72450_a).hashCode();
		code = 31 * code + new Double(field_72448_b).hashCode();
		code = 31 * code + new Double(field_72449_c).hashCode();
		return code;
	}
}
